﻿Public Class Pony_Editor

    'The pony displayed in the preview window and where settings are changed (live).
    Friend Preview_Pony As Pony = Nothing

    'list of the DataGridViews
    Dim grids As New List(Of pony_info_grid)
    Dim loaded = False

    Dim Pony_Name_List As New List(Of String)

    'Keep track of when the grids are being updated when loading a pony
    'otherwise we incorrectly think the user is making changes
    Dim already_updating As Boolean = False

    'keep track of if changes have been made since the last save
    Dim has_saved = True
    'keep track of any saves made at all in the editor
    'if so, we'll need to reload files when we quit.
    Friend changes_made = False


    'Used so we can swap grid positions and keep tract of how everything is sorted when we refresh
    Class pony_info_grid

        Friend grid As DataGridView
        Friend slot As Integer
        Friend button As Button
        Friend Sort_Column As System.Windows.Forms.DataGridViewColumn = Nothing
        Friend Sort_Order As System.Windows.Forms.SortOrder = Nothing

        Sub New(ByVal _grid As DataGridView, ByVal _slot As Integer, ByVal _button As Button)
            grid = _grid
            slot = _slot
            button = _button
        End Sub

    End Class

    Private Sub Pony_Editor_Load(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MyBase.Load

        Try
            Pony_Name_List.Clear()

            has_saved = True
            changes_made = False

            Preview_Pony = Nothing

            loaded = False

            Pony_Behaviors_Grid.Rows.Clear()
            Pony_Effects_Grid.Rows.Clear()
            Pony_Interaction_Grid.Rows.Clear()
            Pony_Speech_Grid.Rows.Clear()

            grids.Clear()
            grids.Add(New pony_info_grid(Pony_Behaviors_Grid, 0, New_Behavior_Button))
            grids.Add(New pony_info_grid(Pony_Speech_Grid, 1, New_Speech_Button))
            grids.Add(New pony_info_grid(Pony_Effects_Grid, 2, New_Effect_Button))
            grids.Add(New pony_info_grid(Pony_Interaction_Grid, 3, New_Interaction_Button))

            Pony_Selection_View.Items.Clear()

            Dim pony_image_list As New ImageList()
            pony_image_list.ImageSize = New Size(50, 50)

            'add all possible ponies to the selection window.
            For Each PonyPanel As Panel In My.Forms.Main.SplitContainer1.Panel1.Controls

                Dim label As Label = Nothing
                Dim pony_picturebox As PictureBox = Nothing

                For Each Control As Control In PonyPanel.Controls
                    If Control.GetType Is GetType(Label) Then
                        If Control.Text <> "How many?" Then
                            label = Control
                        End If
                    End If
                    If Control.GetType Is GetType(PictureBox) Then
                        pony_picturebox = Control
                    End If
                Next

                If IsNothing(label) OrElse IsNothing(pony_picturebox) Then Continue For

                Pony_Name_List.Add(label.Text)

                If IsNothing(pony_picturebox.Image) Then
                    pony_picturebox.Image = My.Resources.Mystery_Thumb
                End If

                pony_image_list.Images.Add(pony_picturebox.Image.Clone())

            Next

            Pony_Selection_View.LargeImageList = pony_image_list
            Pony_Selection_View.SmallImageList = pony_image_list

            Dim pony_menu_order As Integer = 0
            For Each PonyName As String In Pony_Name_List
                Pony_Selection_View.Items.Add(New ListViewItem(PonyName, pony_menu_order))
                pony_menu_order += 1
            Next

            Pony_Selection_View.Columns.Add("Pony")

            loaded = True

        Catch ex As Exception
            MsgBox("Error loading the editor..." & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try

    End Sub

    'get the index of the pony by name in the SelectablePonies list.  
    'This is needed because the order they appear in on the menu is different that in the 
    'list of all selectable ponies, due to filters.
    Private Function GetPonyOrder(ByVal ponyname As String) As Integer

        Dim index As Integer = 0

        For Each Pony As Pony In My.Forms.Main.Selectable_Ponies
            If Pony.Name = ponyname Then
                Return index
            Else
                index += 1
            End If
        Next

        Throw New Exception("Pony not found in selectable pony list.")

    End Function

    Private Sub Pony_Selection_View_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Pony_Selection_View.SelectedIndexChanged

        Try

            If Pony_Selection_View.SelectedItems.Count = 0 Then Exit Sub

            If has_saved = False Then
                If Save_Dialog.ShowDialog() = DialogResult.Cancel Then
                    Exit Sub
                End If
            End If

            If Pony_Selection_View.SelectedIndices.Count = 0 Then Exit Sub

            LoadPony(Pony_Selection_View.SelectedIndices(0))

            has_saved = True

        Catch ex As Exception
            MsgBox("Error selecting pony..." & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Exit Sub
        End Try

    End Sub

    Private Sub LoadPony(ByVal menu_index As Integer)

        My.Forms.Main.Pony_Shutdown(True)

        Dim index As Integer = GetPonyOrder(Pony_Name_List(menu_index))

        'as everywhere else, we make a copy from the master copy in selectable_ponies
        Preview_Pony = My.Forms.Main.Selectable_Ponies(index).Duplicate

        If Preview_Pony.Behaviors.Count = 0 Then
            Preview_Pony.Add_Behavior("Default", 1, 60, 60, 0, My.Resources.Mystery_Thumb, My.Resources.Mystery_Thumb, Pony.Allowed_Moves.None, "", "", "")
        End If

        My.Forms.Main.Active_Ponies.Add(Preview_Pony)

        Load_Parameters(Preview_Pony)

        SaveSortOrder()
        RestoreSortOrder()

        My.Forms.Main.Pony_Startup()

    End Sub

    Private Sub Editor_Closing(ByVal sender As Object, ByVal e As System.Windows.Forms.FormClosingEventArgs) Handles Me.FormClosing
        If has_saved = False Then
            If Save_Dialog.ShowDialog() = DialogResult.Cancel Then
                e.Cancel = True
                Exit Sub
            End If
        End If
    End Sub

    Private Sub Editor_Close(ByVal sender As Object, ByVal e As System.Windows.Forms.FormClosedEventArgs) Handles Me.FormClosed
        My.Forms.Main.Pony_Shutdown(True)
    End Sub

    Friend Function GetPreviewWindow_Location() As Point

        Dim BorderHeight As Integer = SystemInformation.BorderSize.Height
        Dim BorderWidth As Integer = SystemInformation.BorderSize.Width
        Dim TitlebarHeight As Integer = SystemInformation.CaptionHeight + BorderHeight

        Return New Point((Pony_Preview_Panel.Location.X + Me.Location.X + BorderWidth), (Pony_Preview_Panel.Location.Y + Me.Location.Y + TitlebarHeight))

    End Function


    Friend Function GetPreviewWindow_Center() As Point

        Dim location = GetPreviewWindow_Location()
        Return New Point(location.X + Pony_Preview_Panel.Size.Width / 4, location.Y + Pony_Preview_Panel.Size.Height / 4)

    End Function

    Friend Function GetPrewviewWindow_RandomPoint() As Point

        Dim location = GetPreviewWindow_Location()
        Return New Point(location.X + Pony_Preview_Panel.Size.Width * Rnd(), location.Y + Pony_Preview_Panel.Size.Height * Rnd())

    End Function

    'Columns in the various datagridviews

    Enum B_Grid
        Activate_Behavior = 0
        Behavior_Name = 1
        Original_Name = 2
        Group = 3
        Group_name = 4
        Probability = 5
        Max_Duraton = 6
        Min_Duration = 7
        Speed = 8
        R_Image = 9
        L_Image = 10
        Movement = 11
        Start_Speech = 12
        End_Speech = 13
        Follow = 14
        Link = 15
        Link_Order = 16
        Skip_Randomly = 17
        Dont_Repeat_Animations = 18
    End Enum

    Enum E_Grid
        Effect_Name = 0
        Original_Name = 1
        Behavior = 2
        R_Image = 3
        L_Image = 4
        Duration = 5
        Repeat = 6
        R_Loc = 7
        R_Center = 8
        L_Loc = 9
        L_Center = 10
        Follow = 11
        Dont_Repeat_Animations = 12
    End Enum
    Enum S_Grid

        Speech_Name = 0
        Original_Name = 1
        Group = 2
        Group_Name = 3
        Text = 4
        Sound = 5
        Use_Randomly = 6

    End Enum

    Enum I_Grid

        Interaction_Name = 0
        Original_Name = 1
        Probability = 2
        Proximity = 3
        Targets = 4
        Interact_Mode = 5
        Behaviors = 6
        Delay = 7

    End Enum

    'This is used to keep track of the links in each behavior chain.
    Private Structure chain_link

        Dim behavior As Pony.Behavior
        Dim order As Integer
        Dim series As Integer

    End Structure


    Friend Sub Load_Parameters(ByRef pony As Pony)

        Try


            If already_updating = True Then
                Exit Sub
            End If

            already_updating = True

            Pony_Behaviors_Grid.Rows.Clear()
            Pony_Effects_Grid.Rows.Clear()
            Pony_Interaction_Grid.Rows.Clear()
            Pony_Speech_Grid.Rows.Clear()

            Pony_Name_Box.Text = pony.Name

            current_behavior_label.Text = "N/A"
            current_behavior_timeleft_label.Text = "N/A"



            Dim effect_behavior_list As DataGridViewComboBoxColumn = Pony_Effects_Grid.Columns(E_Grid.Behavior)
            effect_behavior_list.Items.Clear()

            Dim linked_behavior_list As DataGridViewComboBoxColumn = Pony_Behaviors_Grid.Columns(B_Grid.Link)
            linked_behavior_list.Items.Clear()
            linked_behavior_list.Items.Add("None")

            Dim start_speech_list As DataGridViewComboBoxColumn = Pony_Behaviors_Grid.Columns(B_Grid.Start_Speech)
            Dim end_speech_list As DataGridViewComboBoxColumn = Pony_Behaviors_Grid.Columns(B_Grid.End_Speech)

            start_speech_list.Items.Clear()
            start_speech_list.Items.Add("None")
            end_speech_list.Items.Clear()
            end_speech_list.Items.Add("None")


            Dim unnamed_counter = 1
            For Each speech As Pony.Behavior.Speaking_Line In pony.Speaking_Lines

                Dim name = speech.Name

                If speech.Name = "Unnamed" Then
                    name = "Unnamed #" & unnamed_counter
                    unnamed_counter += 1
                    speech.Name = name
                End If

                Pony_Speech_Grid.Rows.Add(name, name, speech.Group, pony.getBehaviorGroupName(speech.Group), speech.Text, get_filename(speech.SoundFile), (Not speech.Skip).ToString())
                start_speech_list.Items.Add(LCase(name))
                end_speech_list.Items.Add(LCase(name))
            Next

            For Each behavior In pony.Behaviors
                linked_behavior_list.Items.Add(behavior.Name)
                effect_behavior_list.Items.Add(behavior.Name)
            Next

            'Go through each behavior to see which ones are part of a chain, and if so, which order they go in.
            Dim all_chains As New List(Of chain_link)

            Dim link_series = 0
            For Each behavior In pony.Behaviors

                If (behavior.Linked_Behavior_Name) <> "" AndAlso behavior.Linked_Behavior_Name <> "None" Then

                    Dim not_start_of_chain = False
                    For Each other_behavior In pony.Behaviors
                        If other_behavior.Linked_Behavior_Name = behavior.Name Then
                            not_start_of_chain = True
                            Exit For
                        End If
                    Next

                    'ignore behaviors that are not the first ones in a chain
                    '(chains that loop forever are ignored)
                    If not_start_of_chain Then Continue For

                    Dim new_link As New chain_link
                    new_link.behavior = behavior
                    new_link.order = 1
                    link_series += 1
                    new_link.series = link_series

                    Dim link_order As New List(Of chain_link)
                    link_order.Add(new_link)

                    Dim next_link = behavior.Linked_Behavior_Name

                    Dim depth = 1

                    Dim no_more = False
                    Do Until no_more = True OrElse next_link = "None"
                        link_order = Find_Next_Link(next_link, depth, link_series, pony, link_order)
                        depth = depth + 1
                        If (link_order(link_order.Count - 1).behavior.Linked_Behavior_Name) = "" OrElse link_order.Count <> depth Then
                            no_more = True
                        Else
                            next_link = link_order(link_order.Count - 1).behavior.Linked_Behavior_Name
                        End If
                    Loop

                    For Each link In link_order
                        all_chains.Add(link)
                    Next

                End If
            Next


            For Each behavior In pony.Behaviors

                Dim follow_name As String = ""
                If behavior.follow_object_name <> "" Then
                    follow_name = behavior.follow_object_name
                Else
                    If behavior.destination_xcoord <> 0 AndAlso behavior.destination_ycoord <> 0 Then
                        follow_name = behavior.destination_xcoord & " , " & behavior.destination_ycoord
                    Else
                        follow_name = "Select..."
                    End If
                End If

                Dim link_depth = ""
                For Each link In all_chains
                    If link.behavior.Name = behavior.Name Then
                        If link_depth <> "" Then
                            link_depth += ", " & link.series & "-" & link.order
                        Else
                            link_depth = link.series & "-" & link.order
                        End If

                    End If
                Next

                Dim chance = "N/A"
                If behavior.Skip = False Then
                    chance = (behavior.chance_of_occurance * 100).ToString & "%"
                End If

                With behavior
                    Pony_Behaviors_Grid.Rows.Add("Run", .Name, .Name, .Group, pony.GetBehaviorGroupName(.Group), _
                                        chance, _
                                        .MaxDuration, _
                                        .MinDuration, _
                                        .Speed, _
                                        get_filename(.right_image_path), _
                                        get_filename(.left_image_path), _
                                        Movement_ToString(.Allowed_Movement), _
                                        LCase(.Start_Line_Name), _
                                        LCase(.End_Line_Name), _
                                        follow_name, _
                                        .Linked_Behavior_Name, _
                                        link_depth,
                                        .Skip, .dont_repeat_image_animations)

                End With
            Next

            Dim effects = get_preview_pony_effect_list()

            For Each effect As Pony.Behavior.effect In effects
                With effect
                    Pony_Effects_Grid.Rows.Add(.name, _
                                                 .name, _
                                                 .behavior_name, _
                                                 get_filename(.right_image_path), _
                                                 get_filename(.left_image_path), _
                                                 .Duration, _
                                                 .Repeat_Delay, _
                                                 Location_ToString(.placement_direction_right), _
                                                 Location_ToString(.centering_right), _
                                                 Location_ToString(.placement_direction_left), _
                                                 Location_ToString(.centering_left), _
                                                 .follow, .dont_repeat_image_animations)
                End With
            Next


            For Each Interaction As Pony.Interaction In Preview_Pony.Interactions

                Dim random_all = ""
                If Interaction.Select_All_Targets = True Then
                    random_all = "All"
                Else
                    random_all = "Random"
                End If

                With Interaction
                    Pony_Interaction_Grid.Rows.Add(.Name, _
                                                   .Name, _
                                                   (.Probability * 100).ToString & "%", _
                                                   .Proximity_Activation_Distance, _
                                                   "Select...",
                                                   random_all,
                                                   "Select...",
                                                   .Reactivation_Delay)

                End With


            Next

            'to make sure that speech match behaviors
            Preview_Pony.Link_Behaviors()

            already_updating = False


            Dim Conflicting_names As Boolean = False
            Dim conflicts As New List(Of String)

            For Each behavior In pony.Behaviors
                For Each otherbehavior In pony.Behaviors

                    If ReferenceEquals(behavior, otherbehavior) Then Continue For

                    For Each effect In behavior.Effects
                        For Each othereffect In otherbehavior.Effects

                            If ReferenceEquals(effect, othereffect) Then Continue For

                            If LCase(effect.name) = LCase(othereffect.name) Then
                                Conflicting_names = True
                                If Not conflicts.Contains("Effect: " & effect.name) Then conflicts.Add("Effect: " & effect.name)
                            End If
                        Next
                    Next

                    If LCase(behavior.Name) = LCase(otherbehavior.Name) Then
                        Conflicting_names = True
                        If Not conflicts.Contains("Behavior: " & behavior.Name) Then conflicts.Add("Behavior: " & behavior.Name)
                    End If
                Next

            Next

            For Each speech In pony.Speaking_Lines
                For Each otherspeech In pony.Speaking_Lines
                    If ReferenceEquals(speech, otherspeech) Then Continue For
                    If LCase(speech.Name) = LCase(otherspeech.Name) Then
                        Conflicting_names = True
                        If Not conflicts.Contains("Speech: " & speech.Name) Then conflicts.Add("Speech: " & speech.Name)
                    End If
                Next

            Next

            For Each interaction In pony.Interactions
                For Each otherinteraction In pony.Interactions
                    If ReferenceEquals(interaction, otherinteraction) Then Continue For
                    If LCase(interaction.Name) = LCase(otherinteraction.Name) Then
                        Conflicting_names = True
                        If Not conflicts.Contains("Interaction: " & interaction.Name) Then conflicts.Add("Interaction: " & interaction.Name)
                    End If
                Next

            Next

            If Conflicting_names Then

                Dim conflicts_list As String = ""
                For Each conflict In conflicts
                    conflicts_list += conflict & ControlChars.NewLine
                Next

                MsgBox("Warning: Two or more behaviors, interactions, effects, of speeches have duplicate names.  Please correct these, or the results may be unexpected." & ControlChars.NewLine & conflicts_list)
            End If

        Catch ex As Exception
            MsgBox("Error loading pony parameters! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try

    End Sub

    Private Function Find_Next_Link(ByVal link_name As String, ByVal depth As Integer, ByVal series As Integer, ByRef pony As Pony, ByRef chain_list As List(Of chain_link)) As List(Of chain_link)

        For Each behavior In pony.Behaviors
            If behavior.Name = link_name Then

                depth += 1

                Dim new_link As New chain_link
                new_link.behavior = behavior
                new_link.order = depth
                new_link.series = series

                chain_list.Add(new_link)
                Exit For
            End If
        Next

        Return chain_list

    End Function

    Private Function Movement_ToString(ByVal movement As Pony.Allowed_Moves) As String

        Select Case movement
            Case CByte(0)
                Return "None"
            Case CByte(1)
                Return "Horizontal Only"
            Case CByte(2)
                Return "Vertical Only"
            Case CByte(3)
                Return "Horizontal Vertical"
            Case CByte(4)
                Return "Diagonal Only"
            Case CByte(5)
                Return "Diagonal/horizontal"
            Case CByte(6)
                Return "Diagonal/Vertical"
            Case CByte(7)
                Return "All"
            Case CByte(8)
                Return "MouseOver"
            Case CByte(16)
                Return "Sleep"
            Case CByte(32)
                Return "Dragged"
            Case Else
                Throw New Exception("Invalid movement option: " & movement)
        End Select

    End Function


    Friend Function String_ToMovement(ByVal movement As String) As Pony.Allowed_Moves

        Select Case movement
            Case "None"
                Return Pony.Allowed_Moves.None
            Case "Horizontal Only"
                Return Pony.Allowed_Moves.Horizontal_Only
            Case "Vertical Only"
                Return Pony.Allowed_Moves.Vertical_Only
            Case "Horizontal Vertical"
                Return Pony.Allowed_Moves.Horizontal_Vertical
            Case "Diagonal Only"
                Return Pony.Allowed_Moves.Diagonal_Only
            Case "Diagonal/horizontal"
                Return Pony.Allowed_Moves.Diagonal_Horizontal
            Case "Diagonal/Vertical"
                Return Pony.Allowed_Moves.Diagonal_Vertical
            Case "All"
                Return Pony.Allowed_Moves.All
            Case "MouseOver"
                Return Pony.Allowed_Moves.MouseOver
            Case "Sleep"
                Return Pony.Allowed_Moves.Sleep
            Case "Dragged"
                Return Pony.Allowed_Moves.Dragged
            Case Else
                Throw New Exception("Invalid movement string:" & movement)
        End Select

    End Function

    Private Function Location_ToString(ByVal location As Pony.Directions) As String

        Select Case location

            Case 0
                Return "Top"
            Case 1
                Return "Bottom"
            Case 2
                Return "Left"
            Case 3
                Return "Right"
            Case 4
                Return "Bottom Right"
            Case 5
                Return "Bottom Left"
            Case 6
                Return "Top Right"
            Case 7
                Return "Top Left"
            Case 8
                Return "Center"
            Case 9
                Return "Any"
            Case 10
                Return "Any-Not Center"
            Case Else
                Throw New Exception("Invalid Location/Direction option: " & location)
        End Select
    End Function

    Friend Function string_ToLocation(ByVal location As String) As Pony.Directions
        Select Case location
            Case "Top"
                Return Pony.Directions.top
            Case "Bottom"
                Return Pony.Directions.bottom
            Case "Left"
                Return Pony.Directions.left
            Case "Right"
                Return Pony.Directions.right
            Case "Bottom Right"
                Return Pony.Directions.bottom_right
            Case "Bottom Left"
                Return Pony.Directions.bottom_left
            Case "Top Right"
                Return Pony.Directions.top_right
            Case "Top Left"
                Return Pony.Directions.top_left
            Case "Center"
                Return Pony.Directions.center
            Case "Any"
                Return Pony.Directions.random
            Case "Any-Not Center"
                Return Pony.Directions.random_not_center
            Case Else
                Throw New Exception("Invalid Location/Direction option: " & location)
        End Select

    End Function

    'If we want to run a behavior that has a follow object, we can add it with this.
    Private Function AddPony(ByVal ponyname As String) As Pony

        Try

            For Each Pony In My.Forms.Main.Active_Ponies
                If LCase(Trim(Pony.Name)) = LCase(Trim(ponyname)) Then
                    Return Pony
                End If
            Next

            For Each Pony As Pony In My.Forms.Main.Selectable_Ponies
                If LCase(Trim(Pony.Name)) = LCase(Trim(ponyname)) Then

                    Dim new_pony = Pony.Duplicate()
                    new_pony.Teleport()
                    My.Forms.Main.Active_Ponies.Add(new_pony)
                    Return new_pony
                End If
            Next

            Return Nothing

        Catch ex As Exception
            MsgBox("Error adding pony to editor! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
            Return Nothing
        End Try

        Return Nothing
    End Function

    Private Function AddEffect(ByVal name As String) As Pony.Behavior.effect

        Try

            Dim effects_list = get_effect_list()

            Dim effect As Pony.Behavior.effect = Nothing

            For Each listing In effects_list
                If LCase(Trim(listing.name)) = LCase(Trim(name)) Then
                    effect = listing
                End If
            Next

            If IsNothing(effect) Then Return Nothing

            If effect.duration <> 0 Then
                effect.end_time = Now.AddSeconds(effect.duration)
                effect.Close_On_New_Behavior = False
            End If


            effect.direction = effect.placement_direction_right
            effect.centering = effect.centering_right

            effect.follow = effect.follow
            effect.name = effect.name
            effect.behavior_name = "none"

            effect.location = GetPrewviewWindow_RandomPoint()

            My.Forms.Main.Active_Effects.Add(effect)
            Preview_Pony.Active_Effects.Add(effect)

            effect.LoadImages()

            Return effect

        Catch ex As Exception
            MsgBox("Error adding effect to editor! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
            Return Nothing
        End Try

    End Function

    'Usually thrown when a value entered isn't in the list of allowed values of a dropdown box in a grid.
    Private Sub GridError(ByVal sender As Object, ByVal e As System.Windows.Forms.DataGridViewDataErrorEventArgs) Handles Pony_Behaviors_Grid.DataError, Pony_Effects_Grid.DataError
        Try

            Dim grid As DataGridView = Nothing

            Dim replacement As String = ""

            Select Case sender.name
                Case "Pony_Effects_Grid"
                    grid = Pony_Effects_Grid
                    replacement = ""
                Case "Pony_Behaviors_Grid"
                    grid = Pony_Behaviors_Grid
                    replacement = "None"
                Case Else
                    Throw New Exception("Unhandled error for grid: " & sender.name)
            End Select

            Dim invalid_value = grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value
            grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value = replacement

            MsgBox("Pony.ini file appears to have an invalid line: '" & invalid_value & "' is not valid for column '" & grid.Columns(e.ColumnIndex).HeaderText _
                   & "'" & ControlChars.NewLine & "Details: Column: " & e.ColumnIndex & " Row: " & e.RowIndex & " - " & e.Exception.Message & ControlChars.NewLine & _
                   ControlChars.NewLine & "Value will be reset.")

        Catch ex As Exception
            MsgBox("Error while trying to handle a dataerror! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try

    End Sub


    Private Sub Editor_Resize(ByVal sender As Object, ByVal e As System.EventArgs) Handles Me.Resize

        If Me.loaded = False Then Exit Sub

        Dim slot1 As DataGridView = Nothing
        Dim slot2 As DataGridView = Nothing
        Dim slot3 As DataGridView = Nothing

        For Each entry In grids
            Select Case entry.slot
                Case 1
                    slot1 = entry.grid
                Case 2
                    slot2 = entry.grid
                Case 3
                    slot3 = entry.grid
            End Select
        Next


        slot1.Size = New Size((Me.Size.Width / 3) - 10, slot1.Size.Height)

        slot2.Location = New Point((Me.Size.Width / 3) + 5, slot2.Location.Y)
        slot2.Size = New Size((Me.Size.Width / 3) - 12, slot2.Size.Height)

        slot3.Location = New Point(2 * (Me.Size.Width / 3), slot3.Location.Y)
        slot3.Size = New Size((Me.Size.Width / 3) - 25, slot3.Size.Height)

    End Sub

    Private Sub Pony_Behaviors_Grid_CellContentClick(ByVal sender As System.Object, ByVal e As System.Windows.Forms.DataGridViewCellEventArgs) Handles Pony_Behaviors_Grid.CellClick

        Try

            If e.RowIndex < 0 Then Exit Sub

            SaveSortOrder()

            Dim changed_behavior_name = Pony_Behaviors_Grid.Rows(e.RowIndex).Cells(B_Grid.Original_Name).Value
            Dim changed_behavior As Pony.Behavior = Nothing

            For Each behavior In Preview_Pony.Behaviors
                If behavior.Name = changed_behavior_name Then
                    changed_behavior = behavior
                    Exit For
                End If
            Next

            If IsNothing(changed_behavior) Then
                Load_Parameters(Preview_Pony)
                RestoreSortOrder()
                Exit Sub
            End If

            Dim changes_made_now = False

            Select Case e.ColumnIndex

                Case B_Grid.Activate_Behavior

                    Dim ponies_to_remove As New List(Of Pony)

                    For Each Pony In My.Forms.Main.Active_Ponies
                        If Pony.Name <> Preview_Pony.Name Then
                            ponies_to_remove.Add(Pony)
                        End If
                    Next

                    For Each Pony In ponies_to_remove
                        My.Forms.Main.Active_Ponies.Remove(Pony)
                    Next

                    My.Forms.Main.Active_Effects.Clear()
                    Preview_Pony.Active_Effects.Clear()

                    Dim follow_form = Nothing
                    If changed_behavior.follow_object_name <> "" Then
                        follow_form = AddPony(changed_behavior.follow_object_name)

                        If IsNothing(follow_form) Then
                            follow_form = AddEffect(changed_behavior.follow_object_name)
                        End If

                        If IsNothing(follow_form) Then
                            Throw New Exception("Specified pony/effect to follow, '" & changed_behavior.follow_object_name & _
                                                "' for behavior '" & changed_behavior_name & "' does not exist!")
                        End If
                    End If

                    Preview_Pony.SelectBehavior(changed_behavior)
                    If Not IsNothing(follow_form) Then
                        changed_behavior.follow_object = follow_form
                    End If


                Case B_Grid.R_Image
                    HidePony()
                    Dim new_image_path = Add_Picture(Preview_Pony.Name & " Right Image...")
                    If Not IsNothing(new_image_path) Then
                        changed_behavior.right_image_path = new_image_path
                        'changed_behavior.right_image = Image.FromFile(new_image_path)
                        changed_behavior.UnloadImages()
                        changed_behavior.LoadImages()
                        Image_Size_Check(changed_behavior.RightImage.Size)
                        changes_made_now = True
                    End If
                    ShowPony()
                Case B_Grid.L_Image
                    HidePony()
                    Dim new_image_path = Add_Picture(Preview_Pony.Name & " Left Image...")
                    If Not IsNothing(new_image_path) Then
                        changed_behavior.left_image_path = new_image_path
                        changed_behavior.UnloadImages()
                        changed_behavior.LoadImages()
                        Image_Size_Check(changed_behavior.LeftImage.Size)
                        changes_made_now = True
                    End If
                    ShowPony()
                Case B_Grid.Follow
                    HidePony()
                    Set_Behavior_Follow_Parameters(changed_behavior)
                    changes_made_now = True
                    ShowPony()
            
                Case Else
                    Exit Sub

            End Select

            'If changes_made_now Then
            '    Load_Parameters(Preview_Pony)
            '    RestoreSortOrder()
            'End If

        Catch ex As Exception
            MsgBox("Error altering pony parameters.  Details: " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try

    End Sub

    Private Sub Pony_Behavior_Grid_UserChanged(ByVal sender As Object, ByVal e As System.Windows.Forms.DataGridViewCellEventArgs) Handles Pony_Behaviors_Grid.CellValueChanged

        Try

            If e.RowIndex < 0 Then Exit Sub

            SaveSortOrder()

            Dim new_value = Pony_Behaviors_Grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value

            Dim changed_behavior_name = Pony_Behaviors_Grid.Rows(e.RowIndex).Cells(B_Grid.Original_Name).Value
            Dim changed_behavior As Pony.Behavior = Nothing

            For Each behavior In Preview_Pony.Behaviors
                If behavior.Name = changed_behavior_name Then
                    changed_behavior = behavior
                    Exit For
                End If
            Next

            If IsNothing(changed_behavior) Then
                Load_Parameters(Preview_Pony)
                Exit Sub
            End If

            Try

                Select Case e.ColumnIndex

                    Case B_Grid.Behavior_Name
                        If new_value = "" Then
                            MsgBox("You must give a behavior a name.  It can't be blank.")
                            Pony_Behaviors_Grid.Rows(e.RowIndex).Cells(B_Grid.Behavior_Name).Value = changed_behavior_name
                            Exit Sub
                        End If

                        If new_value = changed_behavior_name Then
                            Exit Sub
                        End If

                        For Each behavior In Preview_Pony.Behaviors
                            If LCase(behavior.Name) = LCase(new_value) Then
                                MsgBox("Behavior names must be unique.  Behavior '" & new_value & "' already exists.")
                                Pony_Behaviors_Grid.Rows(e.RowIndex).Cells(B_Grid.Behavior_Name).Value = changed_behavior_name
                                Exit Sub
                            End If
                        Next
                        changed_behavior.Name = new_value
                        Pony_Behaviors_Grid.Rows(e.RowIndex).Cells(B_Grid.Original_Name).Value = new_value
                    Case B_Grid.Probability
                        changed_behavior.chance_of_occurance = CDbl(Trim(Replace(new_value, "%", ""))) / 100
                    Case B_Grid.Max_Duraton
                        If new_value > 0 Then
                            changed_behavior.MaxDuration = new_value
                        Else
                            Throw New Exception("Maximum Duration must be greater than 0")
                        End If
                    Case B_Grid.Min_Duration
                        If new_value >= 0 Then
                            changed_behavior.MinDuration = new_value
                        Else
                            Throw New Exception("Minumum Duration must be greater than or equal to 0")
                        End If
                    Case B_Grid.Speed
                        changed_behavior.speed = CDbl(new_value)
                    Case B_Grid.Movement
                        changed_behavior.Allowed_Movement = String_ToMovement(new_value)
                    Case B_Grid.Start_Speech

                        If new_value = "None" Then
                            changed_behavior.Start_Line_Name = ""
                        Else
                            changed_behavior.Start_Line_Name = new_value
                        End If


                    Case B_Grid.End_Speech
                        If new_value = "None" Then
                            changed_behavior.End_Line_Name = ""
                        Else
                            changed_behavior.End_Line_Name = new_value
                        End If
                        Preview_Pony.Link_Behaviors()
                    Case B_Grid.Link
                        changed_behavior.Linked_Behavior_Name = new_value
                        Preview_Pony.Link_Behaviors()
                    Case B_Grid.Skip_Randomly
                        changed_behavior.Skip = new_value
                    Case B_Grid.Dont_Repeat_Animations
                        changed_behavior.dont_repeat_image_animations = new_value
                    Case B_Grid.Group
                        If new_value < 0 Then
                            MsgBox("You can't have a group ID less than 0.  Note that 0 is reserved and means that the behavior ignores groups and can run at any time.")
                            Exit Sub
                        End If

                        changed_behavior.Group = new_value
                    Case B_Grid.Group_name
                        If changed_behavior.Group = 0 Then
                            MsgBox("You can't change the name of the 'Any' group.")
                            Exit Sub
                        End If

                        If Preview_Pony.GetBehaviorGroupName(changed_behavior.Group) = "Unnamed" Then
                            Preview_Pony.BehaviorGroups.Add(New Pony.BehaviorGroup(new_value, changed_behavior.Group))
                        Else
                            For Each behaviorgroup In Preview_Pony.BehaviorGroups

                                If behaviorgroup.Name = new_value Then
                                    MsgBox("Error:  That group name already exists under a different ID.")
                                    Exit Sub
                                End If
                            Next

                            For Each behaviorgroup In Preview_Pony.BehaviorGroups
                                If behaviorgroup.Number = changed_behavior.Group Then
                                    behaviorgroup.Name = new_value
                                End If
                            Next

                        End If

                End Select

            Catch ex As Exception
                MsgBox("You entered an invalid value for column '" & Pony_Behaviors_Grid.Columns(e.ColumnIndex).HeaderText & "': " & Pony_Behaviors_Grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value & _
                       ControlChars.NewLine & "Details: " & ex.Message)
            End Try

            Preview_Pony.Link_Behaviors()

            If already_updating = False Then
                'Load_Parameters(Preview_Pony)
                'RestoreSortOrder()
                has_saved = False
            End If

        Catch ex As Exception
            MsgBox("Error altering pony parameters.  Details: " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try

    End Sub

    Private Sub Pony_effects_Grid_UserChanged(ByVal sender As Object, ByVal e As System.Windows.Forms.DataGridViewCellEventArgs) Handles Pony_Effects_Grid.CellValueChanged
        Try

            If e.RowIndex < 0 Then Exit Sub

            SaveSortOrder()

            Dim new_value = Pony_Effects_Grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value

            Dim changed_effect_name = Pony_Effects_Grid.Rows(e.RowIndex).Cells(E_Grid.Original_Name).Value
            Dim changed_effect As Pony.Behavior.effect = Nothing

            For Each effect As Pony.Behavior.effect In get_preview_pony_effect_list()
                If effect.name = changed_effect_name Then
                    changed_effect = effect
                    Exit For
                End If
            Next

            If IsNothing(changed_effect) Then
                Load_Parameters(Preview_Pony)
                Exit Sub
            End If

            Try

                Select Case e.ColumnIndex
                    Case E_Grid.Effect_Name
                        If new_value = "" Then
                            MsgBox("You must give an effect a name.  It can't be blank.")
                            Pony_Effects_Grid.Rows(e.RowIndex).Cells(E_Grid.Effect_Name).Value = changed_effect_name
                            Exit Sub
                        End If

                        If new_value = changed_effect_name Then
                            Exit Sub
                        End If

                        For Each effect In get_effect_list()
                            If LCase(effect.name) = LCase(new_value) Then
                                MsgBox("Effect names must be unique.  Effect '" & new_value & "' already exists.")
                                Pony_Effects_Grid.Rows(e.RowIndex).Cells(E_Grid.Effect_Name).Value = changed_effect_name
                                Exit Sub
                            End If
                        Next

                        changed_effect.name = new_value
                        Pony_Effects_Grid.Rows(e.RowIndex).Cells(E_Grid.Original_Name).Value = new_value
                    Case E_Grid.Behavior
                        For Each behavior In Preview_Pony.Behaviors
                            If behavior.Name = changed_effect.behavior_name Then
                                behavior.Effects.Remove(changed_effect)
                                Exit For
                            End If
                        Next
                        changed_effect.behavior_name = new_value
                        For Each behavior In Preview_Pony.Behaviors
                            If behavior.Name = changed_effect.behavior_name Then
                                behavior.Effects.Add(changed_effect)
                                Exit For
                            End If
                        Next
                    Case E_Grid.Duration
                        changed_effect.duration = new_value
                    Case E_Grid.Repeat
                        changed_effect.repeat_delay = new_value
                    Case E_Grid.R_Loc
                        changed_effect.placement_direction_right = string_ToLocation(new_value)
                    Case E_Grid.L_Loc
                        changed_effect.placement_direction_left = string_ToLocation(new_value)
                    Case E_Grid.R_Center
                        changed_effect.centering_right = string_ToLocation(new_value)
                    Case E_Grid.L_Center
                        changed_effect.centering_left = string_ToLocation(new_value)
                    Case E_Grid.Follow
                        changed_effect.follow = new_value
                    Case E_Grid.Dont_Repeat_Animations
                        changed_effect.dont_repeat_image_animations = new_value
                End Select

            Catch ex As Exception
                MsgBox("You entered an invalid value for column '" & Pony_Effects_Grid.Columns(e.ColumnIndex).HeaderText & "': " & Pony_Effects_Grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value & _
                       ControlChars.NewLine & "Details: " & ex.Message)
            End Try

            Preview_Pony.Link_Behaviors()

            If already_updating = False Then
                'Load_Parameters(Preview_Pony)
                'RestoreSortOrder()
                has_saved = False
            End If
           

        Catch ex As Exception
            MsgBox("Error altering pony parameters.  Details: " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try

    End Sub

    Private Sub Pony_Speech_Grid_UserChanged(ByVal sender As Object, ByVal e As System.Windows.Forms.DataGridViewCellEventArgs) Handles Pony_Speech_Grid.CellValueChanged

        Try

            If e.RowIndex < 0 Then Exit Sub

            SaveSortOrder()

            Dim new_value = Pony_Speech_Grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value

            Dim changed_speech_name = Pony_Speech_Grid.Rows(e.RowIndex).Cells(S_Grid.Original_Name).Value
            Dim changed_speech As Pony.Behavior.Speaking_Line = Nothing

            For Each speech In Preview_Pony.Speaking_Lines
                If speech.Name = changed_speech_name Then
                    changed_speech = speech
                    Exit For
                End If
            Next

            If IsNothing(changed_speech) Then
                Load_Parameters(Preview_Pony)
                Exit Sub
            End If

            Try

                Select Case e.ColumnIndex
                    Case S_Grid.Speech_Name
                        If new_value = "" Then
                            MsgBox("You must give a speech a name, it can't be blank")
                            Pony_Speech_Grid.Rows(e.RowIndex).Cells(S_Grid.Speech_Name).Value = changed_speech_name
                            Exit Sub
                        End If

                        If new_value = changed_speech_name Then
                            Exit Sub
                        End If

                        For Each speechname In Preview_Pony.Speaking_Lines
                            If LCase(speechname.Name) = LCase(new_value) Then
                                MsgBox("Speech names must be unique.  Speech '" & new_value & "' already exists.")
                                Pony_Speech_Grid.Rows(e.RowIndex).Cells(S_Grid.Speech_Name).Value = changed_speech_name
                                Exit Sub
                            End If
                        Next
                        changed_speech.Name = new_value
                        Pony_Speech_Grid.Rows(e.RowIndex).Cells(S_Grid.Original_Name).Value = new_value
                    Case S_Grid.Text
                        changed_speech.Text = new_value
                    Case S_Grid.Use_Randomly
                        changed_speech.Skip = Not (new_value)
                    Case S_Grid.Group
                        changed_speech.Group = new_value
                End Select

            Catch ex As Exception
                MsgBox("You entered an invalid value for column '" & Pony_Speech_Grid.Columns(e.ColumnIndex).HeaderText & "': " & Pony_Speech_Grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value & _
                       ControlChars.NewLine & "Details: " & ex.Message)
                Exit Sub
            End Try

            Preview_Pony.Link_Behaviors()

            If already_updating = False Then
                'Load_Parameters(Preview_Pony)
                'RestoreSortOrder()
                has_saved = False
            End If
          

        Catch ex As Exception
            MsgBox("Error altering pony parameters.  Details: " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try

    End Sub


    Private Sub Pony_Interaction_Grid_UserChanged(ByVal sender As Object, ByVal e As System.Windows.Forms.DataGridViewCellEventArgs) Handles Pony_Interaction_Grid.CellValueChanged

        Try

            If e.RowIndex < 0 Then Exit Sub

            SaveSortOrder()

            Dim new_value = Pony_Interaction_Grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value

            Dim changed_interaction_name = Pony_Interaction_Grid.Rows(e.RowIndex).Cells(I_Grid.Original_Name).Value
            Dim changed_interaction As Pony.Interaction = Nothing

            For Each interaction In Preview_Pony.Interactions
                If interaction.Name = changed_interaction_name Then
                    changed_interaction = interaction
                    Exit For
                End If
            Next

            If IsNothing(changed_interaction) Then
                Load_Parameters(Preview_Pony)
                Exit Sub
            End If

            Try

                Select Case e.ColumnIndex
                    Case I_Grid.Interaction_Name
                        If new_value = "" Then
                            MsgBox("You must give an interaction a name.  It can't be blank.")
                            Pony_Interaction_Grid.Rows(e.RowIndex).Cells(I_Grid.Interaction_Name).Value = changed_interaction_name
                            Exit Sub
                        End If

                        If new_value = changed_interaction_name Then
                            Exit Sub
                        End If

                        For Each Interaction In Preview_Pony.Interactions
                            If LCase(Interaction.Name) = LCase(new_value) Then
                                MsgBox("Interaction with name '" & Interaction.Name & "' already exists for this pony.  Please select another name.")
                                Pony_Interaction_Grid.Rows(e.RowIndex).Cells(I_Grid.Interaction_Name).Value = changed_interaction_name
                                Exit Sub
                            End If
                        Next

                        changed_interaction.Name = new_value
                        Pony_Interaction_Grid.Rows(e.RowIndex).Cells(I_Grid.Original_Name).Value = new_value
                    Case I_Grid.Probability
                        changed_interaction.Probability = CDbl(Trim(Replace(new_value, "%", ""))) / 100
                    Case I_Grid.Proximity
                        changed_interaction.Proximity_Activation_Distance = new_value
                    Case I_Grid.Interact_Mode
                        Select Case new_value
                            Case "Any"
                                changed_interaction.Select_All_Targets = True
                            Case "Random"
                                changed_interaction.Select_All_Targets = False
                            Case Else
                                Throw New Exception("Invalid choice for interaction target selection mode: " & new_value)
                        End Select
                    Case I_Grid.Delay
                        changed_interaction.Reactivation_Delay = CInt(new_value)
                End Select

            Catch ex As Exception
                MsgBox("You entered an invalid value for column '" & Pony_Interaction_Grid.Columns(e.ColumnIndex).HeaderText & "': " & Pony_Interaction_Grid.Rows(e.RowIndex).Cells(e.ColumnIndex).Value & _
                       ControlChars.NewLine & "Details: " & ex.Message)
            End Try

            If already_updating = False Then
                'Load_Parameters(Preview_Pony)
                'RestoreSortOrder()
                has_saved = False
            End If
          
        Catch ex As Exception
            MsgBox("Error altering pony parameters.  Details: " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try

    End Sub

    Friend Function get_effect_list() As List(Of Pony.Behavior.effect)

        Dim effect_list As New List(Of Pony.Behavior.effect)

        For Each pony As Pony In My.Forms.Main.Selectable_Ponies
            For Each behavior As Pony.Behavior In pony.Behaviors
                For Each effect As Pony.Behavior.effect In behavior.Effects
                    effect_list.Add(effect)
                Next
            Next
        Next

        Return effect_list
    End Function

    Private Function get_preview_pony_effect_list()

        Dim effect_list As New List(Of Pony.Behavior.effect)

        For Each behavior As Pony.Behavior In Preview_Pony.Behaviors
            For Each effect As Pony.Behavior.effect In behavior.Effects
                effect_list.Add(effect)
            Next
        Next

        Return effect_list

    End Function

    'Swap posistions of the grids.
    Private Sub Swap0_1_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Swap0_1.Click, Swap1_0.Click
        swap_grids(0, 1, Slot0Label, Slot1Label)
    End Sub
    Private Sub Swap2_0_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Swap2_0.Click
        swap_grids(2, 0, Slot2Label, Slot0Label)
    End Sub
    Private Sub Swap3_0_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Swap3_0.Click
        swap_grids(3, 0, Slot3Label, Slot0Label)
    End Sub

    Private Sub swap_grids(ByVal slot0_number As Integer, ByVal slot1_number As Integer, ByVal label0 As Label, ByVal label1 As Label)

        Dim slot0 As DataGridView = Nothing
        Dim slot1 As DataGridView = Nothing
        Dim button0 As Button = Nothing
        Dim button1 As Button = Nothing

        For Each entry In grids
            Select Case entry.slot
                Case slot0_number
                    slot0 = entry.grid
                    button0 = entry.button
                    entry.slot = slot1_number
                Case slot1_number
                    slot1 = entry.grid
                    entry.slot = slot0_number
                    button1 = entry.button
            End Select
        Next

        If IsNothing(slot0) OrElse IsNothing(slot1) OrElse IsNothing(button0) OrElse IsNothing(button1) Then
            Throw New Exception("Error in swap_grids: did not find right control.")
        End If

        Dim tempsize As Size = slot0.Size
        Dim templocation As Point = slot0.Location
        Dim temp_button_size As Size = button0.Size
        Dim temp_button_loc As Point = button0.Location
        Dim temp_button_anchor = button0.Anchor
        Dim templabel As String = label0.Text
        Dim tempanchor = slot0.Anchor

        slot0.Location = slot1.Location
        slot0.Size = slot1.Size
        slot0.Anchor = slot1.Anchor
        label0.Text = label1.Text
        button0.Size = button1.Size
        button0.Location = button1.Location
        button0.Anchor = button1.Anchor

        slot1.Location = templocation
        slot1.Size = tempsize
        slot1.Anchor = tempanchor
        label1.Text = templabel
        button1.Size = temp_button_size
        button1.Location = temp_button_loc
        button1.Anchor = temp_button_anchor

    End Sub

    Private Sub SaveSortOrder()
        For Each control As pony_info_grid In grids
            control.Sort_Column = control.grid.SortedColumn
            control.Sort_Order = control.grid.SortOrder
        Next
    End Sub

    Private Sub RestoreSortOrder()
        Try

            For Each Control As pony_info_grid In grids
                If IsNothing(Control.Sort_Column) Then Continue For
                Control.grid.Sort(Control.Sort_Column, ConvertSortOrder(Control.Sort_Order))
            Next

        Catch ex As Exception
            MsgBox("Error restore sort order for grid. " & ex.Message & ex.StackTrace)
        End Try
    End Sub


    'The datagridview returns a SortOrder object when you ask it how it is sorted.
    'But, when telling it to sort, you need to specify a ListSortDirection instead, which is different...
    Private Function ConvertSortOrder(ByVal sort As System.Windows.Forms.SortOrder) As System.ComponentModel.ListSortDirection

        Select Case sort
            Case SortOrder.Ascending
                Return System.ComponentModel.ListSortDirection.Ascending
            Case SortOrder.Descending
                Return System.ComponentModel.ListSortDirection.Descending
            Case Else
                Return System.ComponentModel.ListSortDirection.Ascending
        End Select

    End Function


    Private Sub Pause_Pony_Button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Pause_Pony_Button.Click

        Try

            If My.Forms.Main.Ponies_Have_Launched Then
                If Pause_Pony_Button.Text = "Pause Pony" Then
                    Pause_Pony_Button.Text = "Resume Pony"
                    My.Forms.Main.MoveTimer.Stop()
                    My.Forms.Main.MoveTimer.Enabled = False
                Else
                    Pause_Pony_Button.Text = "Pause Pony"
                    My.Forms.Main.MoveTimer.Start()
                    My.Forms.Main.MoveTimer.Enabled = True
                End If
            End If

        Catch ex As Exception
            MsgBox("Error on pause/resume! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try

    End Sub

    Private Sub HidePony()

        Pause_Pony_Button.Text = "Resume Pony"
        My.Forms.Main.MoveTimer.Stop()
        My.Forms.Main.MoveTimer.Enabled = False
        If Not IsNothing(Preview_Pony) Then
            For Each PonyGraphicsForm In My.Forms.Main.PonyForms
                PonyGraphicsForm.Visible = False
            Next

            'Preview_Pony.Form.Visible = False
        End If

    End Sub

    Private Sub ShowPony()
        Pause_Pony_Button.Text = "Pause Pony"
        My.Forms.Main.MoveTimer.Start()
        My.Forms.Main.MoveTimer.Enabled = True

        If Not IsNothing(Preview_Pony) Then

            For Each PonyGraphicsForm In My.Forms.Main.PonyForms
                PonyGraphicsForm.Visible = True
            Next

            'Preview_Pony.Form.Visible = True
        End If
    End Sub

    Private Sub New_Behavior_Button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles New_Behavior_Button.Click
        Try

            If IsNothing(Preview_Pony) Then
                MsgBox("Select a pony or create a new one first,")
                Exit Sub
            End If

            HidePony()
            My.Forms.New_Behavior.ShowDialog()

            ShowPony()
            Preview_Pony.SelectBehavior(Preview_Pony.Behaviors(0))

            Load_Parameters(Preview_Pony)
            has_saved = False

        Catch ex As Exception
            MsgBox("Error creating new behavior! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try

    End Sub


    'Note that we try to rename the file and save the path as all lowercase.
    'This is for compatability with other ports that run on case-sensitive operating systems.
    '(Otherwise they go crazy renaming everything with each update.)
    Friend Function Add_Picture(Optional ByVal text As String = "") As String

        Try

            OpenPictureDialog.Filter = "GIF Files (*.gif)|*.gif|PNG Files (*.png)|*.png|JPG Files (*.jpg)|*.jpg|All Files (*.*)|*.*"
            OpenPictureDialog.FilterIndex = 4
            OpenPictureDialog.InitialDirectory = Preview_Pony.Path_to_Files

            If text = "" Then
                OpenPictureDialog.Title = "Select picture..."
            Else
                OpenPictureDialog.Title = "Select picture for: " & text
            End If

            Dim picture_path = Nothing

            If OpenPictureDialog.ShowDialog() = DialogResult.OK Then
                picture_path = OpenPictureDialog.FileName
            Else
                Return Nothing
            End If

            Dim test_bitmap = Image.FromFile(picture_path)

            Dim new_path = Preview_Pony.Path_to_Files & System.IO.Path.DirectorySeparatorChar & get_filename(picture_path)

            If new_path <> picture_path Then
                Try
                    My.Computer.FileSystem.CopyFile(picture_path, LCase(new_path), True)
                Catch ex As Exception
                    MsgBox("Warning:  Couldn't copy the image file to the pony directory.  If you were trying to use the same image for left and right, you can safely ignore this message. " _
                           & ControlChars.NewLine & "Details: " & ex.Message)
                End Try
            End If

            test_bitmap = Image.FromFile(picture_path)

            Return LCase(new_path)

            has_saved = False

        Catch ex As Exception

            MsgBox("Error loading image.  Details: " & ex.Message)
            Return Nothing

        End Try

    End Function


    Private Sub grid_row_deleting(ByVal sender As Object, ByVal e As DataGridViewRowCancelEventArgs) Handles Pony_Behaviors_Grid.UserDeletingRow, Pony_Effects_Grid.UserDeletingRow, _
                                                                                                            Pony_Interaction_Grid.UserDeletingRow, Pony_Speech_Grid.UserDeletingRow
        Try

            SaveSortOrder()
            Dim grid As DataGridView = Nothing

            Select Case sender.name
                Case "Pony_Effects_Grid"
                    grid = Pony_Effects_Grid
                    Dim todelete As Pony.Behavior.effect = Nothing
                    For Each behavior In Preview_Pony.Behaviors
                        For Each effect In behavior.Effects
                            If effect.name = e.Row.Cells(E_Grid.Effect_Name).Value Then
                                todelete = effect
                                Exit For
                            End If
                        Next
                        If Not IsNothing(todelete) Then
                            behavior.Effects.Remove(todelete)
                            Exit Select
                        End If
                    Next
                Case "Pony_Behaviors_Grid"
                    grid = Pony_Behaviors_Grid
                    If grid.RowCount = 1 Then
                        e.Cancel = True
                        MsgBox("A pony must have at least 1 behavior.  You can't delete the last one.")
                    End If
                    Dim todelete As Pony.Behavior = Nothing
                    For Each behavior In Preview_Pony.Behaviors
                        If e.Row.Cells(B_Grid.Behavior_Name).Value = behavior.Name Then
                            todelete = behavior
                            Exit For
                        End If
                    Next
                    If Not IsNothing(todelete) Then
                        Preview_Pony.Behaviors.Remove(todelete)
                    End If
                Case "Pony_Interaction_Grid"
                    grid = Pony_Interaction_Grid
                    Dim todelete As Pony.Interaction = Nothing
                    For Each interaction In Preview_Pony.Interactions
                        If e.Row.Cells(I_Grid.Interaction_Name).Value = interaction.Name Then
                            todelete = interaction
                            Exit For
                        End If
                    Next
                    If Not IsNothing(todelete) Then
                        Preview_Pony.Interactions.Remove(todelete)
                    End If
                Case "Pony_Speech_Grid"
                    grid = Pony_Speech_Grid
                    Dim todelete As Pony.Behavior.Speaking_Line = Nothing
                    For Each speech In Preview_Pony.Speaking_Lines
                        If e.Row.Cells(S_Grid.Speech_Name).Value = speech.Name Then
                            todelete = speech
                            Exit For
                        End If
                    Next
                    If Not IsNothing(todelete) Then
                        Preview_Pony.Speaking_Lines.Remove(todelete)
                        Preview_Pony.SetLines(Preview_Pony.Speaking_Lines)
                    End If
                Case Else
                    Throw New Exception("Unknown grid when deleteing row: " & sender.name)
            End Select

            has_saved = False

            Preview_Pony.Link_Behaviors()

        Catch ex As Exception
            MsgBox("Error handling row deletion! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try

    End Sub

    Private Sub grid_row_deleted(ByVal sender As Object, ByVal e As DataGridViewRowEventArgs) Handles Pony_Behaviors_Grid.UserDeletedRow, Pony_Effects_Grid.UserDeletedRow, _
                                                                                                           Pony_Interaction_Grid.UserDeletedRow, Pony_Speech_Grid.UserDeletedRow
        'Load_Parameters(Preview_Pony)
        'RestoreSortOrder()

    End Sub

    Private Sub Set_Behavior_Follow_Parameters(ByRef behavior As Pony.Behavior)
        Try

            HidePony()


            My.Forms.Follow_Object.Change_Behavior(behavior)
            My.Forms.Follow_Object.ShowDialog()
            Load_Parameters(Preview_Pony)

            ShowPony()

            has_saved = False
        Catch ex As Exception
            MsgBox("Error setting follow parameters! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try
    End Sub

    Friend Function get_filename(ByVal path As String) As String
        Try

            Dim path_components = Split(path, System.IO.Path.DirectorySeparatorChar)

            'force lowercase for compatibility with other ports (like Browser ponies)
            Return LCase(path_components(UBound(path_components)))

        Catch ex As Exception
            MsgBox("Error getting filename from path! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
            Return Nothing
        End Try

    End Function

    Private Sub Pony_Name_Box_TextChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Pony_Name_Box.TextChanged

        If already_updating = False Then
            Preview_Pony.Name = Pony_Name_Box.Text
            has_saved = False
        End If
      
    End Sub

    Private Sub New_Speech_Button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles New_Speech_Button.Click
        Try
            If IsNothing(Preview_Pony) Then
                MsgBox("Select a pony or create a new one first,")
                Exit Sub
            End If

            HidePony()

            My.Forms.New_Speech.ShowDialog()

            ShowPony()

            Load_Parameters(Preview_Pony)
            has_saved = False
        Catch ex As Exception
            MsgBox("Error creating new speech! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try
    End Sub

    Private Sub New_Effect_Button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles New_Effect_Button.Click
        Try

            If IsNothing(Preview_Pony) Then
                MsgBox("Select a pony or create a new one first,")
                Exit Sub
            End If

            HidePony()

            My.Forms.New_Effect.ShowDialog()

            ShowPony()

            Load_Parameters(Preview_Pony)
            has_saved = False
        Catch ex As Exception
            MsgBox("Error creating new effect! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try
    End Sub

    Private Sub New_Interaction_Button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles New_Interaction_Button.Click
        Try

            If IsNothing(Preview_Pony) Then
                MsgBox("Select a pony or create a new one first,")
                Exit Sub
            End If

            HidePony()
            My.Forms.New_Interaction.Change_Interaction(Nothing)
            My.Forms.New_Interaction.ShowDialog()
            ShowPony()

            Load_Parameters(Preview_Pony)
            has_saved = False

        Catch ex As Exception
            MsgBox("Error creating new interaction! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try
    End Sub

    Private Sub new_pony_button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles new_pony_button.Click

        Try

            If has_saved = False Then
                If Save_Dialog.ShowDialog() = DialogResult.Cancel Then
                    Exit Sub
                End If
            End If

            HidePony()

            Dim previous_pony As Pony = Nothing
            If Not IsNothing(Preview_Pony) Then
                previous_pony = Preview_Pony
            End If

            Preview_Pony = New Pony("new pony", New List(Of Pony.Behavior.Speaking_Line), "", 1, -1, New List(Of Pony.BehaviorGroup)).Duplicate

            'Preview_Pony.Form.Visible = False

            If New_Pony.ShowDialog() = DialogResult.Cancel Then

                If Not IsNothing(previous_pony) Then
                    Preview_Pony = previous_pony
                    ShowPony()
                End If

                Exit Sub
            End If

            changes_made = True
            Me.Close()

        Catch ex As Exception
            MsgBox("Error creating new pony! " & ex.Message & ControlChars.NewLine & ex.StackTrace)
            Me.Close()
        End Try

    End Sub

    Friend Sub SavePony(ByVal path As String)

        Try

            'rebuild the list of ponies, in the original order
            Dim temp_list As New List(Of Pony)
            For Each Pony In My.Forms.Main.Selectable_Ponies
                If Pony.Path_to_Files <> Preview_Pony.Path_to_Files Then
                    temp_list.Add(Pony)
                Else
                    temp_list.Add(Preview_Pony)
                End If
            Next

            My.Forms.Main.Selectable_Ponies = temp_list

            Dim comments As New List(Of String)
            Dim ini_file_path = path & System.IO.Path.DirectorySeparatorChar & "pony.ini"
            If My.Computer.FileSystem.FileExists(ini_file_path) Then
                Dim existing_ini As New System.IO.StreamReader(ini_file_path)

                Do Until existing_ini.EndOfStream
                    Dim line = existing_ini.ReadLine()
                    If Mid(line, 1, 1) = "'" Then
                        comments.Add(line)
                    End If
                    line = Nothing
                Loop
                existing_ini.Close()
            End If

            Dim new_ini As New System.IO.StreamWriter(ini_file_path, False, System.Text.Encoding.UTF8)

            For Each line In comments
                new_ini.WriteLine(line)
            Next

            new_ini.WriteLine("Name," & Preview_Pony.Name)

            Dim tags As String = "Categories,"

            For Each category In Preview_Pony.Tags
                tags += quotes(category) & ","
            Next

            new_ini.WriteLine(Mid(tags, 1, tags.Length - 1)) 'write the tags without the ending comma

            Dim cma = ","

            For Each Group As Pony.BehaviorGroup In Preview_Pony.BehaviorGroups
                new_ini.WriteLine("behaviorgroup," & Group.Number & cma & Group.Name)
            Next

            For Each behavior As Pony.Behavior In Preview_Pony.Behaviors
                new_ini.WriteLine("Behavior," & quotes(behavior.Name) & cma & _
                                  behavior.chance_of_occurance & cma & _
                                  behavior.MaxDuration & cma & _
                                  behavior.MinDuration & cma & _
                                  behavior.Speed & cma & _
                                  quotes(get_filename(behavior.right_image_path)) & cma & _
                                  quotes(get_filename(behavior.left_image_path)) & cma & _
                                  Space_To_Under(Movement_ToString(behavior.Allowed_Movement)) & cma & _
                                  quotes(behavior.Linked_Behavior_Name) & cma & _
                                  quotes(behavior.Start_Line_Name) & cma & _
                                  quotes(behavior.End_Line_Name) & cma & _
                                  behavior.Skip & cma & _
                                  behavior.destination_xcoord & cma & _
                                  behavior.destination_ycoord & cma & _
                                  quotes(behavior.follow_object_name) & cma & _
                                  behavior.Auto_Select_Images_On_Follow & cma & _
                                  behavior.follow_stopped_Behavior_name & cma & _
                                  behavior.follow_moving_behavior_name & cma & _
                                  quotes(behavior.right_image_center.X & cma & behavior.right_image_center.Y) & cma & _
                                  quotes(behavior.left_image_center.X & cma & behavior.left_image_center.Y) & cma & _
                                  behavior.dont_repeat_image_animations & cma & behavior.Group)

            Next

            For Each effect As Pony.Behavior.effect In get_preview_pony_effect_list()

                new_ini.WriteLine("Effect," & _
                                  quotes(effect.name) & cma & _
                                  quotes(effect.behavior_name) & cma & _
                                  quotes(get_filename(effect.right_image_path)) & cma & _
                                  quotes(get_filename(effect.left_image_path)) & cma & _
                                  effect.Duration & cma & _
                                  effect.Repeat_Delay & cma & _
                                  Space_To_Under(Location_ToString(effect.placement_direction_right)) & cma & _
                                  Space_To_Under(Location_ToString(effect.centering_right)) & cma & _
                                  Space_To_Under(Location_ToString(effect.placement_direction_left)) & cma & _
                                  Space_To_Under(Location_ToString(effect.centering_left)) & cma & _
                                  effect.follow & cma & _
                                  effect.dont_repeat_image_animations)

            Next

            For Each speech As Pony.Behavior.Speaking_Line In Preview_Pony.Speaking_Lines

                'For compatibility with 'Browser Ponies', we write an .OGG file as the 2nd option.

                If speech.SoundFile = "" Then
                    new_ini.WriteLine("Speak" & cma & _
                                quotes(speech.Name) & cma & _
                                quotes(speech.Text) & cma & _
                                 cma & _
                                speech.Skip & cma & speech.Group)
                Else
                    new_ini.WriteLine("Speak" & cma & _
                                 quotes(speech.Name) & cma & _
                                 quotes(speech.Text) & cma & _
                                 "{" & quotes(get_filename(speech.SoundFile)) & cma & quotes(Replace(get_filename(speech.SoundFile), ".mp3", ".ogg")) & "}" & cma & _
                                 speech.Skip & cma & speech.Group)
                End If
            Next

            new_ini.Close()

            Try

                Dim interactions_lines As New List(Of String)

                Dim interactions_ini As System.IO.StreamReader = New System.IO.StreamReader(My.Forms.Main.Install_Location & System.IO.Path.DirectorySeparatorChar & "interactions.ini")

                Do Until interactions_ini.EndOfStream
                    Dim line = interactions_ini.ReadLine()

                    Dim name_check = SplitWithQualifiers(line, ",", ControlChars.Quote)

                    If UBound(name_check) > 2 Then
                        If name_check(1) = Preview_Pony.Name Then
                            Continue Do
                        End If
                    End If
                    interactions_lines.Add(line)
                Loop
                interactions_ini.Close()

                interactions_ini.Dispose()

                Dim new_interactions As System.IO.StreamWriter = Nothing

                'we alwas use unicode as characters in other languages will not save properly in the default encoding.
                Try
                    new_interactions = New System.IO.StreamWriter(My.Forms.Main.Install_Location & System.IO.Path.DirectorySeparatorChar & "interactions.ini", False, System.Text.Encoding.UTF8)
                Catch ex As Exception
                    'closing then immediately reopening a file may sometimes fail.
                    'Just try again...
                    Try
                        System.Threading.Thread.Sleep(2000)
                        new_interactions = New System.IO.StreamWriter(My.Forms.Main.Install_Location & System.IO.Path.DirectorySeparatorChar & "interactions.ini", False, System.Text.Encoding.UTF8)
                    Catch ex2 As Exception
                        System.Threading.Thread.Sleep(2000)
                        new_interactions = New System.IO.StreamWriter(My.Forms.Main.Install_Location & System.IO.Path.DirectorySeparatorChar & "interactions.ini", False, System.Text.Encoding.UTF8)
                    End Try

                End Try

                If IsNothing(new_interactions) Then Throw New Exception("Unable to write back to interactions.ini file...")

                For Each line In interactions_lines
                    new_interactions.WriteLine(line)
                Next

                For Each Interaction As Pony.Interaction In Preview_Pony.Interactions


                    Dim behaviors_list As String = ""
                    For Each behavior In Interaction.Behavior_List
                        behaviors_list += quotes(behavior.Name) & ","
                    Next


                    behaviors_list = Mid(behaviors_list, 1, behaviors_list.Length - 1)

                    new_interactions.WriteLine(Interaction.Name & cma & _
                                               quotes(Interaction.PonyName) & cma & _
                                               (Interaction.Probability) & cma & _
                                               (Interaction.Proximity_Activation_Distance) & cma & _
                                               parens(Interaction.Targets_String) & cma & _
                                               Interaction.Select_All_Targets & cma & _
                                               parens(behaviors_list) & cma & _
                                               Interaction.Reactivation_Delay)

                Next


                new_interactions.Close()

            Catch ex As Exception
                Throw New Exception("Failed while updating interactions: " & ex.Message)
            End Try
        Catch ex As Exception
            MsgBox("Unable to save pony! Details: " & ControlChars.NewLine & ex.Message)
            Exit Sub
        End Try

        has_saved = True
        MsgBox("Save completed!")

    End Sub

    Friend Function quotes(ByVal text As String) As String

        Return ControlChars.Quote & text & ControlChars.Quote

    End Function

    Friend Function parens(ByVal text As String) As String

        Return "{" & text & "}"

    End Function

    Private Function Space_To_Under(ByVal text As String) As String
        Return Replace(Replace(text, " ", "_"), "/", "_")
    End Function

    Private Sub Save_Button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Save_Button.Click

        Pause_Pony_Button.Enabled = False
        My.Forms.Main.MoveTimer.Stop()
        My.Forms.Main.MoveTimer.Enabled = False

        changes_made = True
        SavePony(Preview_Pony.Path_to_Files)

        Refresh_Button_Click(Nothing, Nothing)

        My.Forms.Main.MoveTimer.Enabled = True
        My.Forms.Main.MoveTimer.Start()
        Pause_Pony_Button.Enabled = True

    End Sub

    Private Sub Pony_Speech_Grid_CellContentClick(ByVal sender As System.Object, ByVal e As System.Windows.Forms.DataGridViewCellEventArgs) Handles Pony_Speech_Grid.CellClick
        Try

            If e.RowIndex < 0 Then Exit Sub

            SaveSortOrder()

            Dim changed_speech_name = Pony_Speech_Grid.Rows(e.RowIndex).Cells(S_Grid.Original_Name).Value
            Dim changed_speech As Pony.Behavior.Speaking_Line = Nothing

            For Each speech As Pony.Behavior.Speaking_Line In Preview_Pony.Speaking_Lines
                If speech.Name = changed_speech_name Then
                    changed_speech = speech
                    Exit For
                End If
            Next

            If IsNothing(changed_speech) Then
                Load_Parameters(Preview_Pony)
                RestoreSortOrder()
                Exit Sub
            End If

            Dim changes_made_now = False

            Select Case e.ColumnIndex

                Case S_Grid.Sound
                    HidePony()
                    changed_speech.SoundFile = SetSound()
                    changes_made_now = True
                    ShowPony()
                Case Else
                    Exit Sub

            End Select

            If changes_made_now Then
                'Load_Parameters(Preview_Pony)
                'RestoreSortOrder()
                has_saved = False
            End If
         

        Catch ex As Exception
            MsgBox("Error altering pony parameters.  Details: " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try
    End Sub

    Private Sub Pony_Effects_Grid_CellContentClick(ByVal sender As System.Object, ByVal e As System.Windows.Forms.DataGridViewCellEventArgs) Handles Pony_Effects_Grid.CellClick
        Try

            If e.RowIndex < 0 Then Exit Sub

            SaveSortOrder()

            Dim changed_effect_name = Pony_Effects_Grid.Rows(e.RowIndex).Cells(E_Grid.Original_Name).Value
            Dim changed_effect As Pony.Behavior.effect = Nothing

            For Each effect As Pony.Behavior.effect In get_preview_pony_effect_list()
                If effect.name = changed_effect_name Then
                    changed_effect = effect
                    Exit For
                End If
            Next

            If IsNothing(changed_effect) Then
                Load_Parameters(Preview_Pony)
                RestoreSortOrder()
                Exit Sub
            End If

            Dim changes_made_now = False

            Select Case e.ColumnIndex

                Case E_Grid.R_Image
                    HidePony()
                    Dim new_image_path = Add_Picture(changed_effect_name & " Right Image...")
                    If Not IsNothing(new_image_path) Then
                        changed_effect.right_image_path = new_image_path
                        'changed_effect.right_image = Image.FromFile(new_image_path)
                        changed_effect.UnloadImages()
                        changed_effect.LoadImages()
                        changes_made_now = True
                    End If
                    ShowPony()
                Case E_Grid.L_Image
                    HidePony()
                    Dim new_image_path = Add_Picture(changed_effect_name & " Left Image...")
                    If Not IsNothing(new_image_path) Then
                        changed_effect.left_image_path = new_image_path
                        'changed_effect.left_image = Image.FromFile(new_image_path)
                        changed_effect.UnloadImages()
                        changed_effect.LoadImages()
                        changes_made_now = True
                    End If
                    ShowPony()
                Case Else
                    Exit Sub

            End Select

            If changes_made_now Then
                'Load_Parameters(Preview_Pony)
                'RestoreSortOrder()
                has_saved = False
            End If
         

        Catch ex As Exception
            MsgBox("Error altering pony parameters.  Details: " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try
    End Sub

    Private Sub Pony_Interaction_Grid_CellContentClick(ByVal sender As System.Object, ByVal e As System.Windows.Forms.DataGridViewCellEventArgs) Handles Pony_Interaction_Grid.CellClick
        Try

            If e.RowIndex < 0 Then Exit Sub

            SaveSortOrder()

            Dim changed_interaction_name = Pony_Interaction_Grid.Rows(e.RowIndex).Cells(I_Grid.Original_Name).Value
            Dim changed_interaction As Pony.Interaction = Nothing

            For Each interaction As Pony.Interaction In Preview_Pony.Interactions
                If interaction.Name = changed_interaction_name Then
                    changed_interaction = interaction
                    Exit For
                End If
            Next

            If IsNothing(changed_interaction) Then
                Load_Parameters(Preview_Pony)
                RestoreSortOrder()
                Exit Sub
            End If

            Dim changes_made_now = False

            Select Case e.ColumnIndex

                Case I_Grid.Targets
                    HidePony()
                    My.Forms.New_Interaction.Change_Interaction(changed_interaction)
                    My.Forms.New_Interaction.ShowDialog()
                    changes_made_now = True
                    ShowPony()
                Case I_Grid.Behaviors
                    HidePony()
                    My.Forms.New_Interaction.Change_Interaction(changed_interaction)
                    My.Forms.New_Interaction.ShowDialog()
                    changes_made_now = True
                    ShowPony()
                Case Else
                    Exit Sub

            End Select

            If changes_made_now Then
                'Load_Parameters(Preview_Pony)
                'RestoreSortOrder()
                has_saved = False
            End If
        

        Catch ex As Exception
            MsgBox("Error altering pony parameters.  Details: " & ex.Message & ControlChars.NewLine & ex.StackTrace)
        End Try
    End Sub

    Function SetSound() As String

        My.Forms.New_Speech.OpenSoundDialog.Filter = "MP3 Files (*.mp3)|*.mp3"
        My.Forms.New_Speech.OpenSoundDialog.InitialDirectory = My.Forms.Main.Install_Location

        Dim sound_path = Nothing

        If My.Forms.New_Speech.OpenSoundDialog.ShowDialog() = DialogResult.OK Then
            sound_path = My.Forms.New_Speech.OpenSoundDialog.FileName
        End If

        If IsNothing(sound_path) Then
            Return ""
            Exit Function
        End If

        Dim new_path = Preview_Pony.Path_to_Files & System.IO.Path.DirectorySeparatorChar & _
            get_filename(sound_path)

        If new_path <> sound_path Then
            My.Computer.FileSystem.CopyFile(sound_path, new_path, True)
        End If

        Return get_filename(sound_path)

    End Function

    Sub Image_Size_Check(ByVal image_size As Size)

        If image_size.Height > Pony_Preview_Panel.Size.Height OrElse
            image_size.Width > Pony_Preview_Panel.Size.Width Then
            MsgBox("Note:  The selected image is too large for the Pony Editor's preview window.  The results shown will not be accurate, but the pony will still work fine.")
        End If

    End Sub

    Private Sub Edit_Tags_Button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Edit_Tags_Button.Click

        If IsNothing(Preview_Pony) Then
            MsgBox("Select a pony first!")
            Exit Sub
        End If

        HidePony()
        Tags_Form.ShowDialog()
        has_saved = False
        ShowPony()

    End Sub

    Private Sub Set_Image_Centers_Button_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Set_Image_Centers_Button.Click

        If IsNothing(Preview_Pony) Then
            MsgBox("Select a pony first.")
            Exit Sub
        End If
        HidePony()
        Image_Centers_Form.ShowDialog()
        has_saved = False
        ShowPony()

    End Sub

    Private Sub Refresh_Button_Click(sender As System.Object, e As System.EventArgs) Handles Refresh_Button.Click
        Load_Parameters(Preview_Pony)
        RestoreSortOrder()
    End Sub
End Class
