﻿Imports System.Collections.Generic
Imports System.Drawing
Imports System.Globalization
Imports System.IO
Imports System.Threading

Module ImageManagement

    ''' <summary>
    ''' Maintains a set of AnimatedImages, allowing reuse of files from a cache.
    ''' Provides performance options for the loading and caching of images.
    ''' </summary>
    Public Class ImageManager
#Region "AnimatedImageWithReferences struct"
        ''' <summary>
        ''' Represents an AnimatedImage with an additional count of objects maintaining a reference to it.
        ''' </summary>
        Private Structure AnimatedImageWithReferences
            ''' <summary>
            ''' Gets the AnimatedImage being referenced. This can be null if the image has not yet been loaded.
            ''' </summary>
            Public ReadOnly Image As AnimatedImage
            ''' <summary>
            ''' The number of references to this AnimatedImage.
            ''' </summary>
            Public ReadOnly ReferenceCount As Integer

            ''' <summary>
            ''' Initializes a new instance of the AnimatedImageWithReferences struct with references but no image.
            ''' </summary>
            ''' <param name="referenceCount__1">The number of references.</param>
            Public Sub New(referenceCount__1 As Integer)
                Image = Nothing
                ReferenceCount = referenceCount__1
            End Sub

            ''' <summary>
            ''' Initializes a new instance of the AnimatedImageWithReferences struct for an existing AnimatedImage with one reference.
            ''' </summary>
            ''' <param name="image__1">The existing AnimatedImage to which references are required.</param>
            Public Sub New(image__1 As AnimatedImage)
                Image = image__1
                ReferenceCount = 1
            End Sub

            ''' <summary>
            ''' Initializes a new instance of the AnimatedImageWithReferences struct for an existing AnimatedImage with the given number of
            ''' references.
            ''' </summary>
            ''' <param name="image__1">The existing AnimatedImage to which references are required.</param>
            ''' <param name="referenceCount__2">The number of references to this AnimatedImage.</param>
            Private Sub New(image__1 As AnimatedImage, referenceCount__2 As Integer)
                Image = image__1
                ReferenceCount = referenceCount__2
            End Sub

            ''' <summary>
            ''' Returns a new instance with the reference count incremented.
            ''' </summary>
            ''' <param name="current">The current instance from which a new one should be created.</param>
            ''' <returns>A new AnimatedImageWithReferences with the reference count incremented.</returns>
            Public Shared Function IncrementReferenceCount(current As AnimatedImageWithReferences) As AnimatedImageWithReferences
                Return New AnimatedImageWithReferences(current.Image, current.ReferenceCount + 1)
            End Function

            ''' <summary>
            ''' Returns a new instance with the reference count decremented.
            ''' </summary>
            ''' <param name="current">The current instance from which a new one should be created.</param>
            ''' <returns>A new AnimatedImageWithReferences with the reference count decremented.</returns>
            Public Shared Function DecrementReferenceCount(current As AnimatedImageWithReferences) As AnimatedImageWithReferences
                Return New AnimatedImageWithReferences(current.Image, current.ReferenceCount - 1)
            End Function

            ''' <summary>
            ''' Provides a string representing this AnimatedImage and the number of references it possesses.
            ''' </summary>
            ''' <returns>Returns a string representing this AnimatedImage and the number of references it possesses.</returns>
            Public Overrides Function ToString() As String
                Return "{" & Convert.ToString(Image) & " ReferenceCount = " & ReferenceCount & "}"
            End Function

            ''' <summary>
            ''' Releases all resources used by this object.
            ''' </summary>
            Public Sub Dispose()
                If Image IsNot Nothing Then
                    Image.Dispose()
                End If
            End Sub
        End Structure
#End Region

        ''' <summary>
        ''' Gets a value indicating whether an image is loaded only once needed.
        ''' If true, an image will not be loaded until it is first needed, or LoadImages() is called.
        ''' If false, images are loaded when they are added to the collection.
        ''' </summary>
        Public Property OnDemandLoading() As Boolean
            Get
                Return m_OnDemandLoading
            End Get
            Private Set(value As Boolean)
                m_OnDemandLoading = Value
            End Set
        End Property
        Private m_OnDemandLoading As Boolean
        ''' <summary>
        ''' Gets a value indicating whether images are disposed of as soon as they become unreferenced.
        ''' If true, they will remain in the cache should they be needed later, but will occupy memory.
        ''' UnloadImages() can be called to dispose of these images.
        ''' If false, an image will be automatically disposed of once no more objects reference it.
        ''' </summary>
        Public Property CacheUnreferencedImages() As Boolean
            Get
                Return m_CacheUnreferencedImages
            End Get
            Private Set(value As Boolean)
                m_CacheUnreferencedImages = Value
            End Set
        End Property
        Private m_CacheUnreferencedImages As Boolean
        ''' <summary>
        ''' <para>Gets a value indicating whether new instances of animated images will check for duplicate frames
        ''' when generating individual bitmaps for each frame.</para>
        ''' <para>If true, identical frames will only be loaded once, and simply referred to several times.
        ''' This will reduce memory usage where frames are often reused.</para>
        ''' <para>If false, every frame will be generated. If the image is unlikely to contain duplicate frames, loading
        ''' time can be reduced with this setting.</para>
        ''' </summary>
        Public Property IdentifyCommonFrames() As Boolean
            Get
                Return m_IdentifyCommonFrames
            End Get
            Private Set(value As Boolean)
                m_IdentifyCommonFrames = Value
            End Set
        End Property
        Private m_IdentifyCommonFrames As Boolean

        ''' <summary>
        ''' The cache of loaded images. The key is the filename, the value is the image plus a count of references to that image.
        ''' </summary>
        Private images As New Dictionary(Of String, AnimatedImageWithReferences)(100)

        ''' <summary>
        ''' Gets the number of images in cache.
        ''' </summary>
        Public ReadOnly Property ImageCount() As Integer
            Get
                Return images.Count
            End Get
        End Property

        ''' <summary>
        ''' Initializes a new instance of the ImageManager class to manage a
        ''' collection of images in a cache and with given performance options.
        ''' </summary>
        ''' <param name="lazyLoading"> If true, images will be loaded only once a draw is requested or when LoadImages() is called.
        ''' If false, images will be loaded as soon as they are added to the collection.</param>
        ''' <param name="lazyUnloading">If true, images will remain in the cache until UnloadImages() or Dispose() is called.
        ''' If false, images will be unloaded as soon as no object is referencing them.</param>
        ''' <param name="reuseFramesInAnimations">Indicates if identical frames in animated images should be identified, this allows reuse
        ''' of frames and a smaller memory footprint, but will take additional time to initialize.</param>
        Public Sub New(lazyLoading As Boolean, lazyUnloading As Boolean, Optional reuseFramesInAnimations As Boolean = True)
            OnDemandLoading = lazyLoading
            CacheUnreferencedImages = lazyUnloading
            IdentifyCommonFrames = reuseFramesInAnimations
        End Sub

        ''' <summary>
        ''' Increments the number of references to the given filename.
        ''' If the file at the given filename has not yet been referenced in this collection,
        ''' it will be loaded in accordance with the OnDemandLoading property.
        ''' </summary>
        ''' <param name="fileName">The filename of the image whose references should be incremented.</param>
        Public Sub AddReferenceToFile(fileName As String)
            Dim imageWithReferences As AnimatedImageWithReferences = New AnimatedImageWithReferences
            If images.TryGetValue(fileName, imageWithReferences) Then
                ' If the key exists, just increment the reference count.
                images(fileName) = AnimatedImageWithReferences.IncrementReferenceCount(imageWithReferences)
            Else
                ' Create a new object to hold references, load the image right now if requested.
                Dim newImageWithReferences As AnimatedImageWithReferences
                If OnDemandLoading Then
                    newImageWithReferences = New AnimatedImageWithReferences(1)
                Else
                    newImageWithReferences = New AnimatedImageWithReferences(New AnimatedImage(fileName, IdentifyCommonFrames))
                End If
                ' Adds our new image to our collection.
                images.Add(fileName, newImageWithReferences)
            End If
        End Sub

        ''' <summary>
        ''' Decrements the number of references to the given filename.
        ''' If the file at the given filename has no references after this,
        ''' it will be unloaded in accordance with the CacheUnreferencedImages property.
        ''' </summary>
        ''' <param name="fileName">The filename of the image whose references should be decremented.</param>
        Public Sub RemoveReferenceToFile(fileName As String)
            ' We only have stuff to do if the given key exists.
            Dim imageWithReferences As AnimatedImageWithReferences = New AnimatedImageWithReferences
            If images.TryGetValue(fileName, imageWithReferences) Then
                If imageWithReferences.ReferenceCount > 1 OrElse CacheUnreferencedImages Then
                    ' Decrement the reference count if we are keeping the image.
                    images(fileName) = AnimatedImageWithReferences.DecrementReferenceCount(imageWithReferences)
                Else
                    ' There are no references to this image, so we can remove it.
                    images.Remove(fileName)
                    imageWithReferences.Dispose()
                End If
            End If
        End Sub

        ''' <summary>
        ''' Returns the AnimatedImage for the given file, either from cache or after loading from file.
        ''' </summary>
        ''' <param name="fileName">The path to the file for which an AnimatedImage instance is required.</param>
        ''' <returns>An AnimatedImage instance for the given file.</returns>
        Public Function ImageFromFile(fileName As String) As AnimatedImage
            ' Get the image from cache, or else load it on request.
            Dim image As AnimatedImageWithReferences = New AnimatedImageWithReferences
            If Not images.TryGetValue(fileName, image) Then
                image = New AnimatedImageWithReferences(New AnimatedImage(fileName, IdentifyCommonFrames))
                images.Add(fileName, image)
            End If

            Return image.Image
        End Function

        ''' <summary>
        ''' Synchronization object for use when updating progress.
        ''' </summary>
        Private syncObject As New Object()
        ''' <summary>
        ''' Number of images to load (i.e. the number of images not already loaded).
        ''' </summary>
        Private imagesToLoad As Integer
        ''' <summary>
        ''' The number of images loaded by this run of the LoadImages method.
        ''' </summary>
        Private imagesLoaded As Integer
        ''' <summary>
        ''' Contains a list of the exceptions generated for each images that failed to load.
        ''' </summary>
        Private loadingErrors As New LinkedList(Of Exception)()

#Region "LoadImagesArgs struct"
        ''' <summary>
        ''' Arguments to be passed to a thread loading an image.
        ''' </summary>
        Private Structure LoadImagesArgs
            ''' <summary>
            ''' The filename of the image to be loaded by this thread.
            ''' </summary>
            Public ReadOnly Filename As String
            ''' <summary>
            ''' The event used to block threads from modifying the cache until enumeration is finished and all threads are spun up.
            ''' </summary>
            Public ReadOnly CacheAccessible As ManualResetEvent
            ''' <summary>
            ''' The event used to signal that all threads have finished loading their images.
            ''' </summary>
            Public ReadOnly LoadingFinished As ManualResetEvent

            ''' <summary>
            ''' Initializes a new instance of the LoadImagesArgs struct.
            ''' </summary>
            ''' <param name="filename__1">The filename of the image to load.</param>
            ''' <param name="cacheAccessible__2">The event to use to block threads from accessing the cache until it is safe to do so.</param>
            ''' <param name="loadingFinished__3">The event to be signaled once the last images has been loaded.</param>
            Public Sub New(filename__1 As String, cacheAccessible__2 As ManualResetEvent, loadingFinished__3 As ManualResetEvent)
                Filename = filename__1
                CacheAccessible = cacheAccessible__2
                LoadingFinished = loadingFinished__3
            End Sub
        End Structure
#End Region

        ''' <summary>
        ''' Loads the images that have not yet been loaded. Loading operations are done across several threads.
        ''' You can subscribe to the ImageLoaded event whilst this method is running to monitor progress of loading.
        ''' </summary>
        Public Sub LoadImages()
            imagesToLoad = 0

            ' Determine the number of images to be loaded.
            For Each image As AnimatedImageWithReferences In images.Values
                If image.Image Is Nothing Then
                    imagesToLoad += 1
                End If
            Next

            If imagesToLoad > 0 Then
                imagesLoaded = 0
                Dim cacheAccessible As New ManualResetEvent(False)
                Dim loadingFinished As New ManualResetEvent(False)

                Try
                    ' For each unloaded image, send a new item to the pool that will load it.
                    ' Starting the threads now allows loading to begin, but the cache may not be modified until enumeration is complete.
                    For Each image As KeyValuePair(Of String, AnimatedImageWithReferences) In images
                        If image.Value.Image Is Nothing Then
                            ThreadPool.QueueUserWorkItem(New WaitCallback(AddressOf LoadImageOnThread), New LoadImagesArgs(image.Key, cacheAccessible, loadingFinished))
                        End If
                    Next

                    ' Signal that the cache may now be modified.
                    cacheAccessible.[Set]()

                    ' Wait for the signal that we have loaded everything.
                    loadingFinished.WaitOne()

                    ' Return the individual exceptions for each image inside a new exception.
                    If loadingErrors.Count <> 0 Then
                        Dim loadError As New Exception(String.Format(CultureInfo.CurrentCulture, "Failed to load images successfully. {0} images failed to load.", loadingErrors.Count))
                        Dim i As Integer = 0
                        For Each loadingError As Exception In loadingErrors
                            loadError.Data.Add(i, loadingError)
                            i += 1
                        Next
                        loadingErrors.Clear()
                        Throw loadError
                    End If
                Finally
                    'cacheAccessible.Dispose()
                    'loadingFinished.Dispose()
                End Try
            End If
        End Sub

        ''' <summary>
        ''' Loads a new image from file.
        ''' </summary>
        ''' <param name="argsObject">The LoadImagesArguments object containing the filename of the image to load.</param>
        Private Sub LoadImageOnThread(argsObject As Object)
            Dim args As LoadImagesArgs = CType(argsObject, LoadImagesArgs)

            Dim newImageWithReferences As New AnimatedImageWithReferences()
            Dim loadSucceeded As Boolean = True
            Try
                ' Try to load the image, this is the slow part that benefits from multithreading.
                newImageWithReferences = New AnimatedImageWithReferences(New AnimatedImage(args.Filename, IdentifyCommonFrames))
            Catch ex As Exception
                ' Save the exception for later.
                SyncLock loadingErrors
                    loadingErrors.AddLast(ex)
                End SyncLock
                loadSucceeded = False
            Finally
                ' Wait until the cache is no longer being enumerated.
                args.CacheAccessible.WaitOne()

                ' Update the cache.
                SyncLock images
                    If loadSucceeded Then
                        images(args.Filename) = newImageWithReferences
                    Else
                        images.Remove(args.Filename)
                    End If
                End SyncLock
            End Try

            ' Update our progress.
            SyncLock syncObject
                ' Raise the ImageLoaded event.
                imagesToLoad -= 1
                imagesLoaded += 1
                OnImageLoad(New ImageLoadedEventArgs(imagesLoaded, imagesToLoad))

                ' If that was the last image, signal we are finished.
                If imagesToLoad = 0 Then
                    args.LoadingFinished.[Set]()
                End If
            End SyncLock
        End Sub

        ''' <summary>
        ''' Occurs when an image was loaded by the manager during the LoadImages() method.
        ''' </summary>
        Public Event ImageLoaded As EventHandler(Of ImageLoadedEventArgs)

        ''' <summary>
        ''' Raises the ImageLoaded event.
        ''' </summary>
        ''' <param name="e">The ImageLoadedEventArgs that contains the event data.</param>
        Protected Sub OnImageLoad(e As ImageLoadedEventArgs)
            RaiseEvent ImageLoaded(Me, e)
        End Sub

        ''' <summary>
        ''' Unloads the images currently unreferenced in the cache.
        ''' Optionally also clears the whole cache, including referenced images.
        ''' </summary>
        ''' <param name="unloadReferencedImages">If true, unloads all the images from cache.
        ''' If false, unloads only unreferenced images from cache.</param>
        Public Sub UnloadImages(Optional unloadReferencedImages As Boolean = False)
            If Not unloadReferencedImages Then
                ' Generate a list of images with no references.
                Dim keysToRemove As New LinkedList(Of String)()
                For Each image As KeyValuePair(Of String, AnimatedImageWithReferences) In images
                    If image.Value.ReferenceCount <= 0 Then
                        keysToRemove.AddLast(image.Key)
                    End If
                Next

                ' Remove these images.
                For Each key As String In keysToRemove
                    images(key).Dispose()
                    images.Remove(key)
                Next
            Else
                ' Remove all images.
                For Each image As AnimatedImageWithReferences In images.Values
                    image.Dispose()
                Next
                images.Clear()
            End If
        End Sub
    End Class

#Region "ImageLoadedEventArgs class"
    ''' <summary>
    ''' Provides data for the ImageLoaded event.
    ''' </summary>
    Public Class ImageLoadedEventArgs
        Inherits EventArgs
        ''' <summary>
        ''' Gets the number of images loaded.
        ''' </summary>
        Public Property Loaded() As Integer
            Get
                Return m_Loaded
            End Get
            Private Set(value As Integer)
                m_Loaded = Value
            End Set
        End Property
        Private m_Loaded As Integer
        ''' <summary>
        ''' Gets the number of images still to be loaded.
        ''' </summary>
        Public Property Remaining() As Integer
            Get
                Return m_Remaining
            End Get
            Private Set(value As Integer)
                m_Remaining = Value
            End Set
        End Property
        Private m_Remaining As Integer

        ''' <summary>
        ''' Initializes a new instance of the ImageLoadedEventArgs class.
        ''' </summary>
        ''' <param name="loaded__1">The number of images loaded.</param>
        ''' <param name="remaining__2">The number of images still to be loaded.</param>
        Public Sub New(loaded__1 As Integer, remaining__2 As Integer)
            Loaded = loaded__1
            Remaining = remaining__2
        End Sub
    End Class
#End Region

    ''' <summary>
    ''' Provides methods to create and draw animated images based on a time index.
    ''' </summary>
    Public NotInheritable Class AnimatedImage
        Implements IDisposable
        ''' <summary>
        ''' Indicates if we have disposed of this instance.
        ''' </summary>
        Private disposed As Boolean = False
        ''' <summary>
        ''' Gets or sets the palette remapping array.
        ''' </summary>
        Private Shared Property PaletteRemap() As Color(,)
            Get
                Return m_PaletteRemap
            End Get
            Set(value As Color(,))
                m_PaletteRemap = Value
            End Set
        End Property
        Private Shared m_PaletteRemap As Color(,)

        ''' <summary>
        ''' The dimensions of the image.
        ''' </summary>
        Private m_size As Size
        ''' <summary>
        ''' Gets the filename of the image file.
        ''' </summary>
        Public Property FileName() As String
            Get
                Return m_FileName
            End Get
            Private Set(value As String)
                m_FileName = Value
            End Set
        End Property
        Private m_FileName As String
        ''' <summary>
        ''' Gets the width of the image file.
        ''' </summary>
        Public ReadOnly Property Width() As Integer
            Get
                Return m_size.Width
            End Get
        End Property
        ''' <summary>
        ''' Gets the height of the image file.
        ''' </summary>
        Public ReadOnly Property Height() As Integer
            Get
                Return m_size.Height
            End Get
        End Property
        ''' <summary>
        ''' Gets the size of the image file.
        ''' </summary>
        Public ReadOnly Property Size() As Size
            Get
                Return m_size
            End Get
        End Property
        ''' <summary>
        ''' Gets a value indicating whether the image has more than one frame, and thus is animated.
        ''' </summary>
        Public Property IsAnimated() As Boolean
            Get
                Return m_IsAnimated
            End Get
            Private Set(value As Boolean)
                m_IsAnimated = Value
            End Set
        End Property
        Private m_IsAnimated As Boolean

        ''' <summary>
        ''' Gets a value indicating whether duplicate frames were eliminated during creation of this image.
        ''' </summary>
        Public Property CommonFramesIdentified() As Boolean
            Get
                Return m_CommonFramesIdentified
            End Get
            Private Set(value As Boolean)
                m_CommonFramesIdentified = Value
            End Set
        End Property
        Private m_CommonFramesIdentified As Boolean
        ''' <summary>
        ''' Gets a value indicating the total running time of this AnimatedImage, in milliseconds.
        ''' </summary>
        Public Property ImageDuration() As Integer
            Get
                Return m_ImageDuration
            End Get
            Private Set(value As Integer)
                m_ImageDuration = Value
            End Set
        End Property
        Private m_ImageDuration As Integer
        ''' <summary>
        ''' Gets a value indicating how many times this image loops. A value of 0 indicates an endless loop.
        ''' </summary>
        Public Property LoopCount() As Integer
            Get
                Return m_LoopCount
            End Get
            Private Set(value As Integer)
                m_LoopCount = Value
            End Set
        End Property
        Private m_LoopCount As Integer
        ''' <summary>
        ''' Gets the total number of frames in this animation.
        ''' </summary>
        Public Property FrameCount() As Integer
            Get
                Return m_FrameCount
            End Get
            Private Set(value As Integer)
                m_FrameCount = Value
            End Set
        End Property
        Private m_FrameCount As Integer

        ''' <summary>
        ''' Determines if the image has the same duration for all frames.
        ''' </summary>
        Private hasCommonFrameDuration As Boolean = True
        ''' <summary>
        ''' The duration which each frame is shown for (if hasCommonFrameDuration is true).
        ''' </summary>
        Private commonFrameDuration As Integer

        ''' <summary>
        ''' The bitmaps for each frame.
        ''' </summary>
        Private frameBitmaps As List(Of Bitmap)
        ''' <summary>
        ''' Indicates if a bitmap is horizontally flipped.
        ''' </summary>
        Private frameFlipped As List(Of Boolean)
        ''' <summary>
        ''' The duration of each frame in milliseconds.
        ''' </summary>
        Private frameDurations As List(Of Integer)
        ''' <summary>
        ''' Specifies the index in the FrameBitmaps list that the bitmap for
        ''' this frame is contained in (if CommonFramesIdentified is true).
        ''' </summary>
        Private frameIndexes As List(Of Integer)

        ''' <summary>
        ''' Initializes static members of the AnimatedImage class.
        ''' </summary>
        Shared Sub New()
            If My.Forms.Main.OS_Is_Old Then
                PaletteRemap = New Color(0, 1) {{Color.FromArgb(0, 0, 0), Color.FromArgb(1, 1, 1)}}
            End If
        End Sub

        ''' <summary>
        ''' Initializes a new instance of the DesktopPonies.AnimatedImage class from a given file.
        ''' </summary>
        ''' <param name="fileName__1">The path to the file which contains the image to be loaded.</param>
        ''' <param name="identifyCommonFrames">Indicates if identical frames should be identified.
        ''' This allows reuse of frames and a smaller memory footprint, but will take additional time to initialize.</param>
        ''' <exception cref="System.Exception">Thrown if there is an error.</exception>
        Public Sub New(fileName__1 As String, Optional identifyCommonFrames As Boolean = True)
            FileName = fileName__1

            If Path.GetExtension(fileName__1) = ".gif" Then
                AnimatedImageFromGif(identifyCommonFrames)
            Else
                AnimatedImageFromStaticFormat()
            End If
        End Sub

        ''' <summary>
        ''' Initializes a new instance of the DesktopPonies.AnimatedImage class.
        ''' </summary>
        ''' <param name="identifyCommonFrames">Indicates if identical frames should be identified.
        ''' This allows reuse of frames and a smaller memory footprint, but will take additional time to initialize.</param>
        ''' <exception cref="System.Exception">Thrown if there is an error.</exception>
        Private Sub AnimatedImageFromGif(Optional identifyCommonFrames As Boolean = True)
            Try
                CommonFramesIdentified = identifyCommonFrames

                Dim gifImage As GifDecoder
                Using imageStream As New FileStream(FileName, FileMode.Open, FileAccess.Read, FileShare.Read)
                    gifImage = New GifDecoder(imageStream, PaletteRemap)
                End Using

                m_size = gifImage.Size
                LoopCount = gifImage.Iterations

                Dim frameCount__1 As Integer = gifImage.Frames.Count

                ' Create the objects that store the image.
                frameBitmaps = New List(Of Bitmap)(frameCount__1)
                frameFlipped = New List(Of Boolean)(frameCount__1)
                frameDurations = New List(Of Integer)(frameCount__1)
                If identifyCommonFrames Then
                    frameIndexes = New List(Of Integer)(frameCount__1)
                End If

                Dim frameHashes As New List(Of Byte())(frameCount__1)
                For sourceFrame As Integer = 0 To frameCount__1 - 1
                    Dim frameDurationInMilliseconds As Integer = gifImage.Frames(sourceFrame).Duration

                    ' Decoding the gif may have produced frames of zero duration, we can safely drop these.
                    ' If the file has all-zero durations, we're into the land of undefined behavior for animations.
                    ' If we get to the last frame and we have nothing so far, we'll use that just so there is something to display.
                    If frameDurationInMilliseconds = 0 AndAlso Not (ImageDuration = 0 AndAlso sourceFrame = frameCount__1 - 1) Then
                        ' Dispose of unused frame.
                        gifImage.Frames(sourceFrame).Dispose()
                        Continue For
                    End If

                    frameDurations.Add(frameDurationInMilliseconds)
                    ImageDuration += frameDurationInMilliseconds

                    ' Determine if all frames share the same duration.
                    If sourceFrame = 0 Then
                        commonFrameDuration = frameDurationInMilliseconds
                    ElseIf commonFrameDuration <> frameDurationInMilliseconds Then
                        hasCommonFrameDuration = False
                    End If

                    If Not identifyCommonFrames Then
                        ' Add the bitmap to our list of frames.
                        frameBitmaps.Add(gifImage.Frames(sourceFrame).Image)
                        frameFlipped.Add(False)
                    Else
                        ' Calculate the frame hash to check if a duplicate frame exists.
                        ' This will update our collection and given hash list appropriately.
                        CheckForExistingBitmap(gifImage.Frames(sourceFrame), frameHashes)
                    End If
                Next

                frameBitmaps.TrimExcess()
                frameFlipped.TrimExcess()
                If frameIndexes IsNot Nothing Then
                    frameIndexes.TrimExcess()
                End If
                frameDurations.TrimExcess()
                FrameCount = frameDurations.Count
                IsAnimated = FrameCount > 1
            Catch ex As Exception
                Throw New Exception("Could not create a new AnimatedImage from gif file: " & FileName, ex)
            End Try
        End Sub

        ''' <summary>
        ''' Checks if the hash of the given frame image matches any of the given frame hashes.
        ''' If so, the existing bitmap will be reused and lists updated.
        ''' If not, the new frame will be added to the collection.
        ''' </summary>
        ''' <param name="frame">The frame whose image is to be checked.</param>
        ''' <param name="frameHashes">The list of existing frame hashes, this will be updated with a new hash if needed.</param>
        ''' <returns>True if a matching bitmap was found in the given collection, otherwise false.</returns>
        Private Function CheckForExistingBitmap(frame As GifFrame, frameHashes As List(Of Byte())) As Boolean
            ' Get the frame bytes, and calculate the hash value.
            Dim converter As New ImageConverter()
            Dim frameBytes As Byte() = New Byte(-1) {}
            frameBytes = DirectCast(converter.ConvertTo(frame.Image, frameBytes.[GetType]()), Byte())

            Dim frameHash As Byte()
            Using md5 As New System.Security.Cryptography.MD5CryptoServiceProvider()
                frameHash = md5.ComputeHash(frameBytes)
            End Using

            ' Search our existing hashes for a match.
            Dim foundMatchingBitmap As Boolean = False
            Dim i As Integer = 0
            While i < frameHashes.Count AndAlso Not foundMatchingBitmap
                If frameHash.Length = frameHashes(i).Length Then
                    ' Compare hash bytes.
                    Dim foundMatchingHash As Boolean = True
                    For b As Integer = 0 To frameHash.Length - 1
                        If frameHash(b) <> frameHashes(i)(b) Then
                            foundMatchingHash = False
                            Exit For
                        End If
                    Next
                    ' If we found a match, we can reuse the existing bitmap.
                    If foundMatchingHash Then
                        frameIndexes.Add(i)
                        foundMatchingBitmap = True
                        Exit While
                    End If
                End If
                i += 1
            End While

            If Not foundMatchingBitmap Then
                ' If we didn't find any matches, we can add the bitmap to our frames.
                frameBitmaps.Add(frame.Image)
                frameFlipped.Add(False)
                frameHashes.Add(frameHash)
                frameIndexes.Add(frameBitmaps.Count - 1)
            Else
                ' Dispose of duplicate frame.
                frame.Dispose()
            End If

            Return foundMatchingBitmap
        End Function

        ''' <summary>
        ''' Initializes a new instance of the DesktopPonies.AnimatedImage class.
        ''' </summary>
        ''' <exception cref="System.Exception">Thrown if there is an error.</exception>
        Private Sub AnimatedImageFromStaticFormat()
            Try
                CommonFramesIdentified = False

                Using imageStream As New FileStream(FileName, FileMode.Open, FileAccess.Read, FileShare.Read)
                    Using image__1 As Image = Image.FromStream(imageStream)
                        m_size = image__1.Size
                        frameBitmaps = New List(Of Bitmap)(1)
                        frameBitmaps.Add(New Bitmap(image__1))
                    End Using
                End Using

                frameFlipped = New List(Of Boolean)(1)
                frameFlipped.Add(False)
                IsAnimated = False
                LoopCount = 0
                FrameCount = 1
            Catch ex As Exception
                Throw New Exception("Could not create a new AnimatedImage from static image file: " & FileName, ex)
            End Try
        End Sub

        ''' <summary>
        ''' Draws the image onto the given graphics context at the given location.
        ''' The time given will be used to determine the frame shown.
        ''' </summary>
        ''' <param name="target">The graphics surface that will handle drawing.</param>
        ''' <param name="x">The x co-ordinate of the location to draw the top-left corner of the image.</param>
        ''' <param name="y">The y co-ordinate of the location to draw the top-left corner of the image.</param>
        ''' <param name="time">The elapsed time since the initial display of the image, from which a frame will be selected.
        ''' The images looping value will determine how a frame is selected when the time is greater than the animation duration.</param>
        ''' <param name="flip">Indicates if the image should be horizontally flipped from its original direction.</param>
        ''' <param name="scale">The scale to draw the image at.</param>
        ''' <exception cref="System.ObjectDisposedException">Thrown if the object has been disposed.</exception>
        Public Function Draw(target As Graphics, x As Integer, y As Integer, time As TimeSpan, flip As Boolean, scale As Single, dont_repeat_image_animations As Boolean) As Bitmap
            If target Is Nothing Then
                Throw New ArgumentNullException("target")
            End If

            If disposed Then
                Throw New ObjectDisposedException([GetType]().FullName)
            End If

            Dim frame As Integer = 0

            ' Find the frame we need.
            If IsAnimated Then
                ' Get overall time to find in milliseconds.
                Dim timeToSeek As Integer = CInt(Math.Truncate(time.TotalMilliseconds / My.Forms.Options.SlowDownFactor))
                ' Use integer division to find out how many whole loops will run in that time.
                Dim completeLoops As Integer = timeToSeek \ ImageDuration

                If dont_repeat_image_animations AndAlso completeLoops = 0 Then
                    Dim oops = 1
                End If

                If ((LoopCount <> 0) AndAlso completeLoops >= LoopCount) OrElse (dont_repeat_image_animations AndAlso completeLoops > LoopCount) Then
                    ' We have reached the end of looping, and thus want the final frame.
                    frame = frameDurations.Count - 1
                Else
                    ' Subtract the complete loops leaving us with a duration into one run of the animation.
                    Dim durationToSeek As Integer = timeToSeek - (completeLoops * ImageDuration)

                    ' Determine which frame we want.
                    If hasCommonFrameDuration Then
                        frame = durationToSeek \ commonFrameDuration
                    Else
                        While durationToSeek > frameDurations(frame)
                            durationToSeek -= frameDurations(frame)
                            frame += 1
                        End While
                    End If
                End If
            End If

            If frame < 0 Then frame = 0

            Dim index As Integer = If(CommonFramesIdentified, frameIndexes(frame), frame)

            ' If the image is not facing the correct direction according to the caller, flip the frame over.
            If flip <> frameFlipped(index) Then
                frameBitmaps(index).RotateFlip(RotateFlipType.RotateNoneFlipX)
                frameFlipped(index) = Not frameFlipped(index)
            End If

            ' Draw the selected frame.
            If scale = 1.0F Then
                target.DrawImageUnscaled(frameBitmaps(index), x, y)
            Else
                target.DrawImage(frameBitmaps(index), x, y, frameBitmaps(index).Width * scale, frameBitmaps(index).Height * scale)
            End If

            Return frameBitmaps(index)

        End Function

        ''' <summary>
        ''' Releases all resources used by this object.
        ''' </summary>
        Public Sub Dispose() Implements IDisposable.Dispose
            If Not disposed Then
                If frameBitmaps IsNot Nothing Then
                    For Each frame As Bitmap In frameBitmaps
                        frame.Dispose()
                    Next
                End If
                disposed = True
            End If
        End Sub
    End Class

End Module
